package cn.gtmap.gtc.workflow.utils;

import cn.gtmap.gtc.workflow.Constant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

/**
 * @author Fjj
 * @date 2018/8/7
 */
public class DateUtils {

    private static final Logger LOGGER = LoggerFactory.getLogger(DateUtils.class);

    /**
     * Date转Calendar
     *
     * @param date
     * @return
     */
    public static Calendar toCalendar(Date date) {
        if (date == null) {
            return null;
        }
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        return calendar;
    }

    /**
     * String转Calendar
     *
     * @param s
     * @return
     */
    public static Calendar toCalendar(String s, DateFormatType dateFormatType) throws ParseException {
        if (dateFormatType == DateFormatType.DATETIME) {
            SimpleDateFormat dateTimeFormat = new SimpleDateFormat(Constant.DEFAULT_DATETIME_FORMATE);
            return toCalendar(dateTimeFormat.parse(s));
        } else if (dateFormatType == DateFormatType.DATE) {
            SimpleDateFormat dateFormat = new SimpleDateFormat(Constant.DEFAULT_DATE_FORMATE);
            return toCalendar(dateFormat.parse(s));
        }
        return null;
    }

    public enum DateFormatType {

        /**
         * 日期格式类型
         */
        DATE(Constant.DEFAULT_DATE_FORMATE), DATETIME(Constant.DEFAULT_DATETIME_FORMATE);

        String format;

        DateFormatType(String format) {
            this.format = format;
        }
    }

    /**
     * 年月日转Date
     *
     * @param year
     * @param month
     * @param day
     * @return
     */
    public static Date toData(int year, int month, int day) {
        Calendar calendar = Calendar.getInstance();
        calendar.set(year, month - 1, day);
        return calendar.getTime();
    }

    /**
     * Calendar转Date
     *
     * @param calendar
     * @return
     */
    public static Date toData(Calendar calendar) {
        return calendar.getTime();
    }

    /**
     * String转Date
     *
     * @param date
     * @param dateFormatType
     * @return
     */
    public static Date toDate(String date, DateFormatType dateFormatType) throws ParseException {
        if (dateFormatType == DateFormatType.DATETIME) {
            SimpleDateFormat dateTimeFormat = new SimpleDateFormat(Constant.DEFAULT_DATETIME_FORMATE);
            return dateTimeFormat.parse(date);
        } else if (dateFormatType == DateFormatType.DATE) {
            SimpleDateFormat dateFormat = new SimpleDateFormat(Constant.DEFAULT_DATE_FORMATE);
            return dateFormat.parse(date);
        }
        return null;
    }

    /**
     * 计算相差天数
     *
     * @param date1
     * @param date2
     * @return
     */
    public static int calcDayOffset(Date date1, Date date2) {
        Calendar cal1 = Calendar.getInstance();
        cal1.setTime(date1);
        Calendar cal2 = Calendar.getInstance();
        cal2.setTime(date2);
        return calcDayOffset(cal1, cal2);
    }

    /**
     * 计算相差天数
     *
     * @param cal1
     * @param cal2
     * @return
     */
    public static int calcDayOffset(Calendar cal1, Calendar cal2) {
        int day1 = cal1.get(Calendar.DAY_OF_YEAR);
        int day2 = cal2.get(Calendar.DAY_OF_YEAR);

        int year1 = cal1.get(Calendar.YEAR);
        int year2 = cal2.get(Calendar.YEAR);
        if (year1 != year2) {
            int timeDistance = 0;
            for (int i = year1; i < year2; i++) {
                boolean isLeapYear = i % 4 == 0 && i % 100 != 0 || i % 400 == 0;
                if (isLeapYear) {
                    timeDistance += 366;
                } else {
                    timeDistance += 365;
                }
            }
            return timeDistance + (day2 - day1);
        } else {
            return day2 - day1;
        }
    }

    /**
     * 判断是否是同一天
     *
     * @param cal1
     * @param cal2
     * @return
     */
    boolean isSameDay(Calendar cal1, Calendar cal2) {
        return cal1.get(Calendar.MONTH) == cal2.get(Calendar.MONTH) &&
                cal1.get(Calendar.DAY_OF_MONTH) == cal2.get(Calendar.DAY_OF_MONTH) &&
                cal1.get(Calendar.YEAR) == cal2.get(Calendar.YEAR);
    }

    /**
     * String转Time
     *
     * @param strDate
     * @return
     */
    public static java.sql.Time toTime(String strDate) {
        String str = strDate;
        SimpleDateFormat format = new SimpleDateFormat(Constant.DEFAULT_TIME_FORMATE);
        java.util.Date d = null;
        try {
            d = format.parse(str);
            return new java.sql.Time(d.getTime());
        } catch (Exception e) {
            LOGGER.error("日期转换错误", e);
        }
        return null;
    }

    /**
     * 获取时间差
     *
     * @param startTime 开始时间，HH:mm:ss格式时间字符串
     * @param endTime   结束时间，HH:mm:ss格式时间字符串
     * @return 两个时间间隔的分钟数
     */
    public static int getTimeDifferenceInMinutesByTime(String startTime, String endTime) {
        java.sql.Time start = toTime(startTime);
        java.sql.Time end = toTime(endTime);
        if (start != null && end != null) {
            return getTimeDifferenceInMinutesByTime(start, end);
        } else {
            return 0;
        }
    }

    /**
     * 获取时间差
     * 四舍五入 超过30s 按照一分钟计算
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @return 两个时间间隔的分钟数
     */
    public static int getTimeDifferenceInMinutesByTime(java.sql.Time startTime, java.sql.Time endTime) {
        return  Math.round((float)(endTime.getTime() - startTime.getTime()) / (60 * 1000));
    }

    /**
     * 获取时间差s
     * 四舍五入 超过30s 按照一分钟计算
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @return 两个时间间隔的分钟数
     */
    public static int getTimeDifferenceInMinutesByDate(Date startTime, Date endTime) {
        return Math.round((float)(endTime.getTime() - startTime.getTime()) / 60000);
    }

    /**
     * 获取时间差
     *
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @return 两个时间间隔的小时数
     */
    public static int getTimeDifferenceInHoursByDate(Date startTime, Date endTime) {
        return (int) ((endTime.getTime() - startTime.getTime()) / 3600000);
    }

    /**
     * 时间计算，加天数
     *
     * @param calendar
     * @param days
     * @return
     */
    public static Calendar addDays(Calendar calendar, int days) {
        Calendar result = Calendar.getInstance();
        result.setTimeInMillis(calendar.getTimeInMillis());
        result.add(Calendar.DAY_OF_MONTH, days);
        return result;
    }

    /**
     * 加小时数
     *
     * @param calendar
     * @param hours
     * @return
     */
    public static Calendar addHours(Calendar calendar, int hours) {
        Calendar result = Calendar.getInstance();
        result.setTimeInMillis(calendar.getTimeInMillis());
        result.add(Calendar.HOUR_OF_DAY, hours);
        return result;
    }

    /**
     * 加分钟
     *
     * @param calendar
     * @param i
     * @return
     */
    public static Calendar addMinutes(Calendar calendar, int i) {
        Calendar result = Calendar.getInstance();
        result.setTimeInMillis(calendar.getTimeInMillis());
        result.add(Calendar.MINUTE, i);
        return result;
    }
}
