package cn.gtmap.gtc.workflow.helper;

import cn.gtmap.gtc.workflow.Constant;
import cn.gtmap.gtc.workflow.domain.define.WorkDay;
import cn.gtmap.gtc.workflow.utils.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * @author Fjj
 * @date 2018/8/21
 */
public abstract class AbstractWorkdayHelper {

    protected static final Logger LOG = LoggerFactory.getLogger(AbstractWorkdayHelper.class);

    /**
     * 工作日列表（不包含节假日、休息日）
     */
    protected List<WorkDay> workDayList = new LinkedList<>();

    /**
     * 工作日每天上午和下午的工作时长信息
     */
    protected List<WorkMinutes> workMinutesList = new LinkedList<>();

    /**
     * 工作日索引(key为日期格式字符串，value为workDayList中对应索引号)
     */
    protected Map<String, Integer> indexMap = new HashMap<>();

    /**
     * 所有的日期
     */
    protected List<String> allDaysList = new LinkedList<>();

    protected SimpleDateFormat dateFormat = new SimpleDateFormat(Constant.DEFAULT_DATE_FORMATE);

    protected SimpleDateFormat timeFormat = new SimpleDateFormat(Constant.DEFAULT_TIME_FORMATE);

    protected SimpleDateFormat dateTimeFormat = new SimpleDateFormat(Constant.DEFAULT_DATETIME_FORMATE);

    protected static final String A_BLANK = " ";

    /**
     * 记录每天的日期及在allDaysList中索引号以及类型信息
     */
    protected class DayIndexData {

        private String date;

        private int index;

        private int type;

        public DayIndexData(String date, int index, int type) {
            this.date = date;
            this.index = index;
            this.type = type;
        }

        public String getDate() {
            return date;
        }

        public void setDate(String date) {
            this.date = date;
        }

        public int getIndex() {
            return index;
        }

        public void setIndex(int index) {
            this.index = index;
        }

        public int getType() {
            return type;
        }

        public void setType(int type) {
            this.type = type;
        }
    }

    /**
     * 记录当天早晨和下午各自的工作时长
     */
    protected class WorkMinutes {

        /**
         * 造成工作时长（分钟）
         */
        private int morningMinutes;

        /**
         * 下午工作时长（分钟）
         */
        private int afternoonMinutes;

        public WorkMinutes() {

        }

        public WorkMinutes(int morningMinutes, int afternoonMinutes) {
            this.morningMinutes = morningMinutes;
            this.afternoonMinutes = afternoonMinutes;
        }

        public int getMorningMinutes() {
            return morningMinutes;
        }

        public void setMorningMinutes(int morningMinutes) {
            this.morningMinutes = morningMinutes;
        }

        public int getAfternoonMinutes() {
            return afternoonMinutes;
        }

        public void setAfternoonMinutes(int afternoonMinutes) {
            this.afternoonMinutes = afternoonMinutes;
        }
    }

    /**
     * 工作日时间区间类型枚举
     */
    protected enum WorkDayTimeType {

        /**
         * 区间类型
         */
        NOT_WORKDAY(-1, "非工作日"),
        BEFORE_MORNING_STRAT(0, "早晨工作时间以前"),
        MORNING_WORKTIME(1, "早晨工作时间内"),
        LUNCH_BREAK(2, "午休时间"),
        AFTERNOON_WORKTIME(3, "下午工作时间内"),
        AFTER_AFTERNOON_END(4, "下午下班以后");

        /**
         * 值
         */
        private int value;

        /**
         * 描述
         */
        private String remark;

        WorkDayTimeType(int value, String remark) {
            this.value = value;
            this.remark = remark;
        }

    }

    /**
     * 初始化workDayList、indexMap、allDaysTypeMap
     *
     * @param dates
     */
    public void init(List<WorkDay> dates) {
        workDayList.clear();
        workMinutesList.clear();
        indexMap.clear();
        allDaysList.clear();
        int index = 0;
        for (WorkDay workDay : dates) {
            //初始化工作日信息（不包含节假日和休息日），并进行索引
            if (workDay.getDayType() == 0) {
                workDayList.add(workDay);
                workMinutesList.add(new WorkMinutes(DateUtils.getTimeDifferenceInMinutesByTime(workDay.getMorningTimeStart(), workDay.getMorningTimeEnd()),
                        DateUtils.getTimeDifferenceInMinutesByTime(workDay.getAfternoonTimeStart(), workDay.getAfternoonTimeEnd())));
                indexMap.put(workDay.getWorkDay(), index++);
            }
            //初始化所有日期信息，并索引
            allDaysList.add(workDay.getWorkDay());
        }
    }

    /**
     * 检查工作时间
     *
     * @param calendar
     * @return
     */
    public WorkDayTimeType checkWorkTimeType(Calendar calendar) {
        if (!checkWorkDay(calendar)) {
            return WorkDayTimeType.NOT_WORKDAY;
        }
        Date date = calendar.getTime();
        String timeStr = timeFormat.format(date);
        WorkDay workDay = workDayList.get(indexMap.get(dateFormat.format(date)));
        if (timeStr.compareTo(workDay.getMorningTimeStart()) < 0) {
            return WorkDayTimeType.BEFORE_MORNING_STRAT;
        } else if (timeStr.compareTo(workDay.getMorningTimeStart()) >= 0 && timeStr.compareTo(workDay.getMorningTimeEnd()) < 0) {
            return WorkDayTimeType.MORNING_WORKTIME;
        } else if (timeStr.compareTo(workDay.getMorningTimeEnd()) >= 0 && timeStr.compareTo(workDay.getAfternoonTimeStart()) < 0) {
            return WorkDayTimeType.LUNCH_BREAK;
        } else if (timeStr.compareTo(workDay.getAfternoonTimeStart()) >= 0 && timeStr.compareTo(workDay.getAfternoonTimeEnd()) < 0) {
            return WorkDayTimeType.AFTERNOON_WORKTIME;
        } else {
            return WorkDayTimeType.AFTER_AFTERNOON_END;
        }
    }

    /**
     * 检查是否是工作日
     *
     * @param calendar 传入需要验证的日期
     * @return true工作日，false节假日或周末
     */
    public boolean checkWorkDay(Calendar calendar) {
        return indexMap.containsKey(dateFormat.format(calendar.getTime()));
    }

    /**
     * 检查是否是工作日
     *
     * @param str 传入需要验证的日期,yyyy-MM-dd格式
     * @return true工作日，false节假日或周末
     */
    public boolean checkWorkDay(String str) {
        return indexMap.containsKey(str);
    }

    /**
     * 根据时间查询当日的工作日配置信息
     *
     * @param calendar
     * @return
     */
    protected WorkDay getWorkDay(Calendar calendar) {
        Integer index = indexMap.get(dateFormat.format(calendar.getTime()));
        return isNull(index) ? null : workDayList.get(index);
    }

    /**
     * 获取某个时间最近的工作时间
     * （起算时间不在工作时间范围内（如节假日、休息日、其它休息时间）则查找最近的工作时间，
     * 如果起算时间本身在工作时间则返回当前传入的起算时间）
     *
     * @param calendar        起算时间
     * @param workDayTimeType 当前起算时间与工作时间的关系
     * @return
     */
    protected Calendar getNearWorkTime(Calendar calendar, WorkDayTimeType workDayTimeType) throws ParseException {
        if (workDayTimeType == null) {
            workDayTimeType = checkWorkTimeType(calendar);
        }
        Date date = calendar.getTime();
        String dateStr = dateFormat.format(date);
        WorkDay workDay;
        Calendar result = null;
        String temp = "";
        switch (workDayTimeType) {
            case NOT_WORKDAY:
                //下一个工作日的早晨开始时间
                workDay = getNextWorkDay(dateStr);
                temp = workDay.getWorkDay() + A_BLANK + workDay.getMorningTimeStart();
                result = DateUtils.toCalendar(dateTimeFormat.parse(temp));
                break;
            case BEFORE_MORNING_STRAT:
                //早晨工作时间前，返回早晨开始时间
                workDay = workDayList.get(indexMap.get(dateStr));
                temp = dateStr + A_BLANK + workDay.getMorningTimeStart();
                result = DateUtils.toCalendar(dateTimeFormat.parse(temp));
                break;
            case MORNING_WORKTIME:
                //早晨工作时间范围内，直接返回
                result = calendar;
                break;
            case LUNCH_BREAK:
                //午休时间，返回下午开始时间
                workDay = workDayList.get(indexMap.get(dateStr));
                temp = dateStr + A_BLANK + workDay.getAfternoonTimeStart();
                result = DateUtils.toCalendar(dateTimeFormat.parse(temp));
                break;
            case AFTERNOON_WORKTIME:
                //下午工作时间范围内，直接返回
                result = calendar;
                break;
            case AFTER_AFTERNOON_END:
                //下午下班后，下一个工作日的早晨开始时间
                temp = getNextWorkDay(dateStr).getWorkDay() + A_BLANK + getNextWorkDay(dateStr).getMorningTimeStart();
                result = DateUtils.toCalendar(dateTimeFormat.parse(temp));
                break;
            default:
        }
        return result;
    }

    /**
     * 获取接下来的工作日的配置信息
     *
     * @param date 示例：2018-08-01
     * @return
     */
    protected WorkDay getNextWorkDay(String date) {
        //workDayList是排序好的，因此只要大于当前输入的日期就是下一个工作日
        for (WorkDay workDay : workDayList) {
            if (workDay.getWorkDay().compareTo(date) > 0) {
                return workDay;
            }
        }
        return null;
    }

    /**
     * 工作日计算，加天数（排除节假日和休息日）,用于计算到期时间
     *
     * @param calendar 起算时间
     * @param day      相加天数，大于0
     * @return Calendar 返回相加day天，并且排除节假日和周末后的日期
     * @throws ParseException 日期转换异常
     */
    public abstract Calendar addDateByWorkDay(Calendar calendar, int day) throws ParseException;

    /**
     * 工作日计算，加分钟数（排除节假日和休息日）,用于计算到期时间
     *
     * @param calendar 起算时间
     * @param minute   相加分钟数
     * @return
     */
    public Calendar addMinuteByWorkDay(Calendar calendar, int minute) throws ParseException {
        int totalMinutes = minute;
        //获取起算时间最近的工作时间，用作真正的起算时间
        Calendar start = getNearWorkTime(calendar, null);
        //获取起算时间当天的剩余工作时间
        WorkDayTimeType workDayTimeType = checkWorkTimeType(start);
        int minutes = getRemainningMinutes(start, workDayTimeType);
        String dateStr = dateFormat.format(start.getTime());
        WorkDay workDay = workDayList.get(indexMap.get(dateStr));
        while (totalMinutes > minutes) {
            //时间不足，到下一天补
            workDay = getNextWorkDay(dateStr);
            //获取工作时间信息
            WorkMinutes workMinutes = workMinutesList.get(indexMap.get(workDay.getWorkDay()));
            minutes += workMinutes.getMorningMinutes() + workMinutes.getAfternoonMinutes();
            dateStr = workDay.getWorkDay();
        }
        //以上步骤只能算到天，下面开始计算当天的具体时间
        int temp = minutes - totalMinutes;
        //判断是在上午还是下午
        WorkMinutes workMinutes = workMinutesList.get(indexMap.get(dateStr));
        if (temp > workMinutes.getAfternoonMinutes()) {
            //在上午
            return DateUtils.addMinutes(DateUtils.toCalendar(dateTimeFormat.parse(dateStr + A_BLANK + workDay.getMorningTimeEnd())), -(temp - workMinutes.getAfternoonMinutes()));
        } else {
            //在下午
            return DateUtils.addMinutes(DateUtils.toCalendar(dateTimeFormat.parse(dateStr + A_BLANK + workDay.getAfternoonTimeEnd())), -temp);
        }
    }

    /**
     * 工作日计算，加小时数（排除节假日和休息日）,用于计算到期时间
     *
     * @param calendar 起算时间
     * @param hour     相加小时数
     * @return
     */
    public Calendar addHourByWorkDay(Calendar calendar, int hour) throws ParseException {
        int totalMinutes = hour * 60;
        return addMinuteByWorkDay(calendar, totalMinutes);
    }

    /**
     * 获取当天剩余工作时间(分钟数)
     *
     * @param start           起算时间，必须是工作日
     * @param workDayTimeType 当前起算时间与工作时间的关系
     * @return
     */
    private int getRemainningMinutes(Calendar start, WorkDayTimeType workDayTimeType) {
        int result = 0;
        WorkDay workDay = getWorkDay(start);
        String timeStr = timeFormat.format(start.getTime());
        switch (workDayTimeType) {
            case BEFORE_MORNING_STRAT:
                //当天早晨工作时间以前，返回当天工作时间总合
                result = (int) (60 * workDay.getDuraction());
                break;
            case MORNING_WORKTIME:
                //早晨的工作时间，返回早晨剩余时间+下午工作时间
                result = (int) ((DateUtils.toTime(workDay.getAfternoonTimeEnd()).getTime()
                        - DateUtils.toTime(workDay.getAfternoonTimeStart()).getTime()) / (60 * 1000));
                result += (int) ((DateUtils.toTime(workDay.getMorningTimeEnd()).getTime()
                        - DateUtils.toTime(timeStr).getTime()) / (60 * 1000));
                break;
            case LUNCH_BREAK:
                //午休时间，返回下午所有工作时间
                result = (int) ((DateUtils.toTime(workDay.getAfternoonTimeEnd()).getTime()
                        - DateUtils.toTime(workDay.getAfternoonTimeStart()).getTime()) / (60 * 1000));
                break;
            case AFTERNOON_WORKTIME:
                //下午工作时间，返回下午剩余时间
                result = (int) ((DateUtils.toTime(workDay.getAfternoonTimeEnd()).getTime()
                        - DateUtils.toTime(timeStr).getTime()) / (60 * 1000));
                break;
            case AFTER_AFTERNOON_END:
                //下午下班以后，返回0
                result = 0;
                break;
            default:
                //特殊情况
                result = -1;
        }
        return result;
    }

    /**
     * 获取两个时间相差多少工作日
     *
     * @param start 起算时间
     * @param now 待算超期的时间，可能在起算时间以前，也可能是以后
     * @return
     */
    public int getBetweenDays(Calendar start, Calendar now){
        int result = 0;
        boolean isTimeout = (now.compareTo(start) > 0);
        if (!isTimeout) {
            //未超时情况下，交换岂止时间便于计算
            Calendar temp = start;
            start = now;
            now = temp;
        }

        String startDayStr = dateFormat.format(start.getTime());
        String endDayStr = dateFormat.format(now.getTime());
        if (startDayStr.equals(endDayStr)) {
            return result;
        }
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(start.getTime());
        calendar.add(Calendar.DATE, 1);
        String tempDayStr = dateFormat.format(calendar.getTime());
        while (startDayStr.compareTo(tempDayStr) < 0) {
            if (checkWorkDay(tempDayStr)) {
                result += 1;
            }
            if (tempDayStr.equals(endDayStr)) {
                break;
            }
            startDayStr = tempDayStr;
            calendar.add(Calendar.DATE, 1);
            tempDayStr = dateFormat.format(calendar.getTime());
        }
        if (isTimeout) {
            return result;
        } else {
            return -result;
        }

    }

    private int getBetweenOnDayMins(Calendar start) throws ParseException {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(start.getTime());
        calendar.set(Calendar.HOUR_OF_DAY, 23);
        calendar.set(Calendar.MINUTE, 59);
        return getTimeOutMinutesInSameDay(start, calendar);
    }

    /**
     * 获取两个时间相差多少工作时长
     *
     * @param start 起算时间
     * @param now 待算超期的时间，可能在起算时间以前，也可能是以后
     * @return
     */
    public int getBetweenHours(Calendar start, Calendar now) throws ParseException {
        int result = 0;
        boolean isTimeout = (now.compareTo(start) > 0);
        if (!isTimeout) {
            //未超时情况下，交换岂止时间便于计算
            Calendar temp = start;
            start = now;
            now = temp;
        }

        String startDayStr = dateFormat.format(start.getTime());
        String endDayStr = dateFormat.format(now.getTime());
        if (checkWorkDay(startDayStr) && startDayStr.equals(endDayStr)) {
            result += getTimeOutMinutesInSameDay(start, now);
        } else {
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(start.getTime());
            while (!checkWorkDay(startDayStr)) {
                calendar.add(Calendar.DATE, 1);
                calendar.set(Calendar.HOUR_OF_DAY, 0);
                calendar.set(Calendar.MINUTE, 0);
                calendar.set(Calendar.SECOND, 0);
                if (calendar.compareTo(now) > 0){
                    return 0;
                }
                startDayStr = dateFormat.format(calendar.getTime());
            }

            while (!startDayStr.equals(endDayStr)) {
                if (checkWorkDay(startDayStr)) {
                    result += getBetweenOnDayMins(calendar);
                }
                calendar.add(Calendar.DATE, 1);
                calendar.set(Calendar.HOUR_OF_DAY, 0);
                calendar.set(Calendar.MINUTE, 0);
                calendar.set(Calendar.SECOND, 0);
                startDayStr = dateFormat.format(calendar.getTime());
            }

            if (checkWorkDay(startDayStr)) {
                result += getTimeOutMinutesInSameDay(calendar, now);
            }
        }
        if (result < 60) {
            return 0;
        }

        result = (int) Math.ceil(result / 60.0f);
        if (isTimeout) {
            return result;
        } else {
            return -result;
        }

    }

    public int getBetweenMinis(Calendar start, Calendar now) throws ParseException {
        int result = 0;
        boolean isTimeout = (now.compareTo(start) > 0);
        if (!isTimeout) {
            //未超时情况下，交换岂止时间便于计算
            Calendar temp = start;
            start = now;
            now = temp;
        }

        String startDayStr = dateFormat.format(start.getTime());
        String endDayStr = dateFormat.format(now.getTime());
        if (checkWorkDay(startDayStr) && startDayStr.equals(endDayStr)) {
            result += getTimeOutMinutesInSameDay(start, now);
        } else {
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(start.getTime());
            while (!checkWorkDay(startDayStr)) {
                calendar.add(Calendar.DATE, 1);
                calendar.set(Calendar.HOUR_OF_DAY, 0);
                calendar.set(Calendar.MINUTE, 0);
                calendar.set(Calendar.SECOND, 0);
                if (calendar.compareTo(now) > 0){
                    return 0;
                }
                startDayStr = dateFormat.format(calendar.getTime());
            }

            while (!startDayStr.equals(endDayStr)) {
                if (checkWorkDay(startDayStr)) {
                    result += getBetweenOnDayMins(calendar);
                }
                calendar.add(Calendar.DATE, 1);
                calendar.set(Calendar.HOUR_OF_DAY, 0);
                calendar.set(Calendar.MINUTE, 0);
                calendar.set(Calendar.SECOND, 0);
                startDayStr = dateFormat.format(calendar.getTime());
            }

            if (checkWorkDay(startDayStr)) {
                result += getTimeOutMinutesInSameDay(calendar, now);
            }
        }
        if (result <= 0) {
            result = 0 - result;
        }

        return result;
    }

    /**
     * 获取超时天数
     *
     * @param due 起算时间，必然是工作日
     * @param now 待算超期的时间，可能在起算时间以前，也可能是以后
     * @return
     */
    public int getTimeOutDays(Calendar due, Calendar now) {
        boolean isTimeout = (now.compareTo(due) > 0);
        if (!isTimeout) {
            //未超时情况下，交换岂止时间便于计算
            Calendar temp = due;
            due = now;
            now = temp;
        }
        String startDayStr = dateFormat.format(due.getTime());
        String endDayStr = dateFormat.format(now.getTime());
        if (startDayStr.equals(endDayStr)) {
            //同一天只要超时就算1天
            return isTimeout ? 1 : 0;
        }
        //不同天之间间隔多少天就算超时多少天
        int result = 0;
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(due.getTime());
        while (startDayStr.compareTo(endDayStr) < 0) {
            if (checkWorkDay(startDayStr)) {
                result += 1;
            }
            calendar.add(Calendar.DATE, 1);
            startDayStr = dateFormat.format(calendar.getTime());
        }
        if (isTimeout) {
            return result;
        } else {
            return result == 0 ? -1 : -result;
        }
    }

    /**
     * 获取下一天
     *
     * @param date
     * @param dateFormatType
     * @return
     * @throws ParseException
     */
    private String getNextDayStr(String date, DateUtils.DateFormatType dateFormatType) throws ParseException {
        Calendar calendar = DateUtils.toCalendar(date, dateFormatType);
        calendar.add(Calendar.DATE, 1);
        return dateFormat.format(calendar.getTime());
    }

    /**
     * 计算同一天的超时时间
     *
     * @param due
     * @param now
     * @return
     */
    public int getTimeOutMinutesInSameDay(Calendar due, Calendar now) throws ParseException {
        //判断是否超期
        Calendar start = due;
        Calendar end = now;
        if (due.compareTo(now) > 0){
            end = due;
            start = now;
        }
        Date startDate = start.getTime();
        Date endDate = end.getTime();
        String dayStr = dateFormat.format(endDate);
        int index = indexMap.get(dayStr);
        WorkDay workDay = workDayList.get(index);
        Date morningStart = dateTimeFormat.parse(dayStr + A_BLANK + workDay.getMorningTimeStart());
        Date morningEnd = dateTimeFormat.parse(dayStr + A_BLANK + workDay.getMorningTimeEnd());
        Date afternoonStart = dateTimeFormat.parse(dayStr + A_BLANK + workDay.getAfternoonTimeStart());
        Date afternoonEnd = dateTimeFormat.parse(dayStr + A_BLANK + workDay.getAfternoonTimeEnd());

        if (startDate.compareTo(morningStart) <= 0){
            startDate = morningStart;
        } else if (startDate.compareTo(morningEnd) >= 0 && startDate.compareTo(afternoonStart) <= 0){
            startDate = afternoonStart;
        } else if (startDate.compareTo(afternoonEnd) > 0){
            startDate = afternoonEnd;
        }

        if (endDate.compareTo(morningStart) <= 0){
            endDate = morningStart;
        } else if (endDate.compareTo(morningEnd) >= 0 && endDate.compareTo(afternoonStart) <= 0){
            endDate = afternoonStart;
        } else if (endDate.compareTo(afternoonEnd) > 0){
            endDate = afternoonEnd;
        }
        int result = 0;
        result = DateUtils.getTimeDifferenceInMinutesByDate(startDate, endDate);

        if (startDate.compareTo(morningEnd) <= 0 && endDate.compareTo(afternoonStart) >= 0){
            result = result - DateUtils.getTimeDifferenceInMinutesByDate(morningEnd, afternoonStart);
        }

        if (result <= 0) {
            return 0;
        } else {
            return result;
        }
    }

    protected boolean isNull(Object object) {
        return object == null;
    }


    public Date buildDueTime(Date dueTime, Date optTime){
        String dataStr = dateFormat.format(optTime).concat(" ").concat(timeFormat.format(dueTime));
        try {
            return dateTimeFormat.parse(dataStr);
        } catch (ParseException e) {
            return dueTime;
        }
    }
}
