package cn.gtmap.gtc.workflow.utils;

import cn.gtmap.gtc.workflow.Constant;
import org.apache.commons.collections.CollectionUtils;
import org.apache.poi.hssf.usermodel.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Field;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;

/**
 * Excel工具类
 *
 * @author Fjj
 * @date 2018/8/30
 */
public class ExcelUtils {

    private static final Logger LOGGER = LoggerFactory.getLogger(ExcelUtils.class);

    private ExcelUtils() {

    }

    /**
     * 导出Excel
     *
     * @param wb        HSSFWorkbook对象
     * @param sheetName sheet名称
     * @param title     标题
     * @param values    内容
     * @param adaptive  自适应列宽
     * @return
     */
    public static HSSFWorkbook buildXlsWorkBook(HSSFWorkbook wb, String sheetName, String[] title, String[][] values, boolean adaptive) {
        // 第一步，创建一个HSSFWorkbook，对应一个Excel文件
        if (wb == null) {
            wb = new HSSFWorkbook();
        }
        // 第二步，在workbook中添加一个sheet,对应Excel文件中的sheet
        HSSFSheet sheet = wb.createSheet(sheetName);
        // 第三步，在sheet中添加表头第0行,注意老版本poi对Excel的行数列数有限制
        HSSFRow row = sheet.createRow(0);
        // 第四步，创建单元格，并设置值表头格式
        HSSFCellStyle style = wb.createCellStyle();
        style.setAlignment(HSSFCellStyle.ALIGN_CENTER);
        HSSFFont font = wb.createFont();
        font.setFontName("黑体");
        font.setBoldweight(HSSFFont.BOLDWEIGHT_BOLD);
        font.setFontHeightInPoints((short) 11);
        style.setFont(font);
        //声明列对象
        HSSFCell cell = null;
        //创建标题
        for (int i = 0; i < title.length; i++) {
            cell = row.createCell(i);
            cell.setCellValue(title[i]);
            cell.setCellStyle(style);
        }
        //创建内容
        if (values != null) {
            for (int i = 0; i < values.length; i++) {
                row = sheet.createRow(i + 1);
                for (int j = 0; j < values[i].length; j++) {
                    //将内容按顺序赋给对应的列对象
                    row.createCell(j).setCellValue(values[i][j]);
                }
            }
        }
        if (adaptive) {
            for (int i = 0; i < title.length; i++) {
                sheet.autoSizeColumn((short) i);
            }
        }
        return wb;
    }

    public static HSSFWorkbook buildXlsWorkBook(HSSFWorkbook wb, String sheetName, String[] title, String[] fieldNames, List<Object> list, boolean adaptive) {
        if (CollectionUtils.isEmpty(list)) {
            return buildXlsWorkBook(wb, sheetName, title, null, adaptive);
        }
        String[][] arr = new String[list.size()][fieldNames.length];
        int i = 0;
        for (Object obj : list) {
            String[] sarr = new String[fieldNames.length];
            for (String fieldName : fieldNames) {
                String value = "";
                Class<?> clazz = obj.getClass();
                Field field = getFieldInClass(clazz, fieldName);
                if (field != null) {
                    field.setAccessible(true);
                    try {
                        value = convertToString(field, obj);
                    } catch (IllegalAccessException e) {
                        LOGGER.error("字段转成String类型时发生错误", e);
                    }
                }
                sarr[i++] = value;
            }
            arr[i++] = sarr;
        }

        return buildXlsWorkBook(wb, sheetName, title, arr, adaptive);
    }

    /**
     * 转换Object中特定字段到String
     *
     * @param field
     * @param obj
     * @return
     * @throws IllegalAccessException
     */
    private static String convertToString(Field field, Object obj) throws IllegalAccessException {
        if (field.getType().isPrimitive()) {
            return String.valueOf(field.get(obj));
        }
        String fieldClassName = field.getType().getName();
        if (fieldClassName.equals(String.class.getName()) || fieldClassName.equals(Short.class.getName()) ||
                fieldClassName.equals(Integer.class.getName()) || fieldClassName.equals(Long.class.getName()) ||
                fieldClassName.equals(Float.class.getName()) || fieldClassName.equals(Double.class.getName()) ||
                fieldClassName.equals(Number.class.getName()) || fieldClassName.equals(Character.class.getName())) {
            return String.valueOf(field.get(obj));
        } else if (fieldClassName.equals(Date.class.getName())) {
            SimpleDateFormat datetimeFormat = new SimpleDateFormat(Constant.DEFAULT_DATETIME_FORMATE);
            return datetimeFormat.format(field.get(obj));
        } else if (fieldClassName.equals(java.sql.Date.class.getName())) {
            SimpleDateFormat dateFormat = new SimpleDateFormat(Constant.DEFAULT_DATE_FORMATE);
            return dateFormat.format(field.get(obj));
        } else if (fieldClassName.equals(java.sql.Time.class.getName())) {
            SimpleDateFormat timeFormat = new SimpleDateFormat(Constant.DEFAULT_TIME_FORMATE);
            return timeFormat.format(field.get(obj));
        } else {
            return obj.toString();
        }
    }

    /**
     * 根据属性名获取属性元素，包括各种安全范围和所有父类
     *
     * @param clazz
     * @param fieldName
     * @return
     */
    private static Field getFieldInClass(Class clazz, String fieldName) {
        Field field = null;
        for (; clazz != Object.class && field == null; clazz = clazz.getSuperclass()) {
            try {
                field = clazz.getDeclaredField(fieldName);
            } catch (Exception e) {
                LOGGER.error("在类中查找指定字段时发生错误", e);
            }
        }
        return field;
    }

}
