package cn.gtmap.gtc.workflow.utils;

import cn.gtmap.gtc.workflow.Constant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * @author zhouk
 * @Title: WorkDayUtil
 * @ProjectName workflow
 * @Description: 节假日工作时间处理
 * @date 2018/7/613:48
 */
public class WorkDayUtil {

    private static final Logger LOGGER = LoggerFactory.getLogger(WorkDayUtil.class);

    private static final String ERROR_DATE_FORMAT = "日期格式不正确";

    private static final long DAY_TO_MILLISECOND = 24 * 60 * 60 * 1000;

    public enum ComputeMode {

        /**
         * 工作日计算种类划分为按天、按小时和不计算三种
         */
        DAY(2, "按天计算", "D"), HOUR(1, "按小时计算", "H"), NONE(0, "不计算", "N");

        ComputeMode(int value, String remark, String name) {
            this.value = value;
            this.remark = remark;
            this.name = name;
        }

        private int value;

        private String remark;

        private String name;

        public int getValue() {
            return value;
        }

        public String getRemark() {
            return remark;
        }

        public String getName() {
            return name;
        }

        public static ComputeMode getByValue(int value) {
            for (ComputeMode computeMode : ComputeMode.values()) {
                if (computeMode.value == value) {
                    return computeMode;
                }
            }
            return NONE;
        }

        public static ComputeMode getByName(String name) {
            for (ComputeMode computeMode : ComputeMode.values()) {
                if (computeMode.name.equals(name)) {
                    return computeMode;
                }
            }
            return NONE;
        }
    }

    /**
     * 获取时间和添加天数后的日期
     *
     * @param day 日期字符串例如 2015-3-10
     * @param num 需要增加的天数例如 7
     * @return
     */
    public static String getDateByAddDays(String day, int num) {
        DateFormat df = new SimpleDateFormat(Constant.DEFAULT_DATE_FORMATE);
        Date nowDate;
        try {
            nowDate = df.parse(day);
        } catch (ParseException e) {
            LOGGER.error(ERROR_DATE_FORMAT, e);
            return null;
        }
        //如果需要向后计算日期 -改为+
        Date newDate2 = new Date(nowDate.getTime() + num * DAY_TO_MILLISECOND);
        return df.format(newDate2);
    }

    /**
     * 获取时间和减少天数后的日期
     *
     * @param day 日期字符串例如 2015-3-10
     * @param num 需要增加的天数例如 7
     * @return
     */
    public static String getDateByDelDays(String day, int num) {
        DateFormat df = new SimpleDateFormat(Constant.DEFAULT_DATE_FORMATE);
        Date nowDate = null;
        try {
            nowDate = df.parse(day);
        } catch (ParseException e) {
            LOGGER.error(ERROR_DATE_FORMAT, e);
            return null;
        }
        //如果需要向后计算日期 -改为+
        Date newDate2 = new Date(nowDate.getTime() - num * DAY_TO_MILLISECOND);
        return df.format(newDate2);
    }

    public static Map<String, Object> getDueDateNum(Date dueDate) {
        DateFormat df = new SimpleDateFormat(Constant.DEFAULT_DATE_FORMATE);
        Map<String, Object> result = new HashMap<>(2);
        //标准超期的时间（天数）
        Date dayDate = null;
        //标准超期的时间（小时数）
        Date hourDate = null;
        try {
            dayDate = df.parse("2000-01-01");
            hourDate = df.parse("2100-01-01");
        } catch (ParseException e) {
            LOGGER.error(ERROR_DATE_FORMAT, e);
            return null;
        }
        if (dueDate.getTime() > hourDate.getTime()) {
            //按小时计算
            result.put("type", ComputeMode.HOUR);
            int num = (int) ((dueDate.getTime() - hourDate.getTime()) / DAY_TO_MILLISECOND);
            result.put("num", num);
        } else {
            //按天计算
            result.put("type", ComputeMode.DAY);
            int num = (int) ((dueDate.getTime() - dayDate.getTime()) / DAY_TO_MILLISECOND);
            result.put("num", num);
        }
        return result;
    }

    /**
     * 根据超期时间获取期限的天数或者期限的小时
     *
     * @param dueDate 超期时间
     * @return
     */
    public static Map<String, Object> getDueDateNum(String dueDate) {
        DateFormat df = new SimpleDateFormat(Constant.DEFAULT_DATE_FORMATE);
        try {
            return getDueDateNum(df.parse(dueDate));
        } catch (ParseException e) {
            LOGGER.error(ERROR_DATE_FORMAT, e);
            return null;
        }
    }

    public static Date getNextReckonTime(Date date){
        if (null == date) {
            date = new Date();
        }
        Calendar reckonTime = DateUtils.toCalendar(date);
        // 时
        //reckonTime.set(Calendar.HOUR_OF_DAY, 0);
        // 分
        reckonTime.set(Calendar.MINUTE, 0);
        // 秒
        reckonTime.set(Calendar.SECOND, 0);
        // 毫秒
        reckonTime.set(Calendar.MILLISECOND, 0);

        return reckonTime.getTime();
    }

}
