package cn.gtmap.gtc.storage.clients.v1;

import cn.gtmap.gtc.common.http.result.PageResult;
import cn.gtmap.gtc.storage.domain.dto.BaseResultDto;
import cn.gtmap.gtc.storage.domain.dto.MultipartDto;
import cn.gtmap.gtc.storage.domain.dto.StorageDto;
import org.springframework.cloud.netflix.feign.FeignClient;
import org.springframework.data.domain.Pageable;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;

/**
 * @author <a href="mailto:yangyang@gtmap.cn">yangyang</a>
 * @version 1.0.2018/9/27
 * @description
 */

@FeignClient(name = "${app.services.storage-app:storage-app}")
@RequestMapping("/storage/rest/v1/storages")
public interface StorageClient {

    /**
     * @param  clientId ， 应用Id，
     * @param  spaceId ， 存储空间Id，
     * @param  owner 拥有者，
     * @param  name 文件名称，
     * @param  nodeId: 上级文件夹Id，
     * @return
     * @description 新建文件夹， nodeId 为空将建在用户根目录下
     */
    @PostMapping("/folder")
    StorageDto createFolder(@RequestParam(name = "clientId", required = false) String clientId,
                            @RequestParam(name = "spaceId", required = false) String spaceId,
                            @RequestParam(name = "nodeId", required = false) String nodeId,
                            @RequestParam(name = "name") String name,
                            @RequestParam(name = "owner") String owner);

    /**
     * 新建文件夹, 项目模式下spaceId不能为空,表示项目Id;names为逗号分隔字符串
     * @param clientId
     * @param spaceId
     * @param nodeId
     * @param names
     * @param owner
     * @return
     */
    @PostMapping("/multi/folder")
    List<StorageDto> createFolderMulti(@RequestParam(name = "clientId", required = false) String clientId,
                                       @RequestParam(name = "spaceId", required = false) String spaceId,
                                       @RequestParam(name = "nodeId", required = false) String nodeId,
                                       @RequestParam(name = "names") String names,
                                       @RequestParam(name = "owner", required = false) String owner);

    /**
     * @param  clientId ， 应用Id，
     * @param  spaceId ， 存储空间Id，
     * @param  owner 拥有者， name 文件名称，
     * @return
     * @description 新建根目录下文件夹, 已存在将不创建
     */
    @PostMapping("/root/folder")
    StorageDto createRootFolder(@RequestParam(name = "clientId", required = false) String clientId,
                                   @RequestParam(name = "spaceId", required = false) String spaceId,
                                   @RequestParam(name = "name") String name,
                                   @RequestParam(name = "owner", required = false) String owner);

    /**
     * 新建根目录下文件夹, 已存在将不创建;names为逗号分隔字符串
     * @param  clientId ， 应用Id，
     * @param  spaceId ， 存储空间Id，
     * @param  owner 拥有者，
     * @param names 文件名称 逗号间隔，
     * @return
     */
    @PostMapping("/root/multi/folder")
    List<StorageDto> createRootFolderMulti(@RequestParam(name = "clientId", required = false) String clientId,
                                                  @RequestParam(name = "spaceId", required = false) String spaceId,
                                                  @RequestParam(name = "names") String names,
                                                  @RequestParam(name = "owner", required = false) String owner);

    /**
     * @param  clientId ， 应用Id，
     * @param  spaceId ， 存储空间Id，
     * @param  nodeId 目录ID，
     * @param  enabled: 0: 删除状态， 1：正常状态，
     * @param  type: 0：目录 1：文件，2: 图片 3： 文档 4： 视频 5：音乐，6：其他
     * @return
     * @description 查询目录菜单
     */
    @GetMapping("/menus")
    List<StorageDto> queryMenus(@RequestParam(name = "clientId", required = false) String clientId,
                                       @RequestParam(name = "spaceId", required = false) String spaceId,
                                       @RequestParam(name = "nodeId", required = false) String nodeId,
                                       @RequestParam(name = "enabled", required = false) Integer enabled,
                                       @RequestParam(name = "type", required = false) Integer type,
                                       @RequestParam(name = "queryCount", required = false) Integer queryCount);

    /**
     * @param  clientId ， 应用Id，
     * @param  spaceId ， 存储空间Id，
     * @param  owner 拥有者， name 文件名称，
     * @param  nodeId: 上级文件夹Id，
     * @param  type: 0：目录 1：文件，2: 图片 3： 文档 4： 视频 5：音乐，6：其他
     * @return
     * @description 查看是否存在
     */
    @GetMapping("/existence")
    boolean checkExist(@RequestParam(name = "clientId", required = false) String clientId,
                       @RequestParam(name = "spaceId", required = false) String spaceId,
                       @RequestParam(name = "nodeId", required = false) String nodeId,
                       @RequestParam(name = "name") String name,
                       @RequestParam(name = "owner", required = false) String owner,
                       @RequestParam(name = "type", required = false) Integer type);

    /**
     * @param id 文件id,  enabled: 0: 删除状态， 1：正常状态
     * @return
     * @description 更该文件状态
     */
    @PostMapping("/status")
    boolean changeStatus(@RequestParam(name = "id") String id, @RequestParam(name = "enabled") Integer enabled);


    /**
     * @param id 文件id,  name: 文件名
     * @return
     * @description 文件重命名
     */
    @PostMapping("/rename/{id}")
    boolean rename(@PathVariable(name = "id") String id, @RequestParam(name = "name") String name);

    /**
     * @param ids 文件id列表
     * @return
     * @description 删除文件
     */
    @DeleteMapping
    boolean deleteStorages(@RequestBody List<String> ids);

    /**
     * @param ids 文件
     * @return
     * @description 根据Ids查询文件信息
     */
    @PostMapping("/list")
    List<StorageDto> listStoragesByIds(@RequestBody List<String> ids);

    /**
     * @param id 文件
     * @return
     * @description 根据Id查询文件信息
     */
    @GetMapping("/{id}")
    StorageDto findById(@PathVariable(name = "id") String id);


    /**
     * @param  clientId ， 应用Id，
     * @param  spaceId ， 存储空间Id，
     * @param  owner 拥有者， name 文件名称，
     * @param  enabled: 0: 删除状态， 1：正常状态，
     * @param  type: 0：目录 1：文件，2: 图片 3： 文档 4： 视频 5：音乐，6：其他
     * @return
     * @description 查询存储空间下所有根节点文件
     */
    @GetMapping("/root")
    PageResult<StorageDto> listRootStorages(Pageable pageable,
                                                    @RequestParam(name = "clientId", required = false) String clientId,
                                                    @RequestParam(name = "spaceId", required = false) String spaceId,
                                                    @RequestParam(name = "owner", required = false) String owner,
                                                    @RequestParam(name = "name", required = false) String name,
                                                    @RequestParam(name = "enabled", required = false) Integer enabled,
                                                    @RequestParam(name = "type", required = false) Integer type);

    /**
     * @param  clientId ， 应用Id，
     * @param  spaceId ， 存储空间Id，
     * @param owner 为空时查询所有用户文件
     * @param name 文件名称
     * @param enabled: 0: 删除状态， 1：正常状态
     * @param type: 0：目录 1：文件，2: 图片 3： 文档 4： 视频 5：音乐，6：其他, null:全部
     * @return
     * @description 按文件类型查询所有文件
     */
    @GetMapping("/mime")
    PageResult<StorageDto> listMimeStorages(Pageable pageable,
                                                        @RequestParam(name = "clientId", required = false) String clientId,
                                                        @RequestParam(name = "spaceId", required = false) String spaceId,
                                                        @RequestParam(name = "owner", required = false) String owner,
                                                        @RequestParam(name = "name", required = false) String name,
                                                        @RequestParam(name = "enabled", required = false) Integer enabled,
                                                        @RequestParam(name = "type", required = false) Integer type);

    /**
     * @param spaceId 空间id
     * @param owner 为空时查询所有用户文件
     * @param name 文件名称
     * @param enabled: 0: 删除状态， 1：正常状态
     * @param type: 0：目录 1：文件，2: 图片 3： 文档 4： 视频 5：音乐，6：其他, null:全部
     * @return
     * @description 查询空间下文件列表
     * @deprecated as of 1.2.2, in favor of using {@link StorageClient#listRootStorages} clientId替换spaceId, spaceId替换porId
     */
    @GetMapping("/project/root")
    @Deprecated
    PageResult<StorageDto> listProjectStorages(Pageable pageable,
                                                    @RequestParam(name = "spaceId", required = false) String spaceId,
                                                    @RequestParam(name = "proId", required = false) String proId,
                                                    @RequestParam(name = "owner", required = false) String owner,
                                                    @RequestParam(name = "name", required = false) String name,
                                                    @RequestParam(name = "enabled", required = false) Integer enabled,
                                                    @RequestParam(name = "type", required = false) Integer type);


    /**
     * @param id:要查询的文件节点，
     * @param name 文件名称，
     * @param enabled: 0: 删除状态， 1：正常状态
     * @param type: 0：目录 1：文件，2: 图片 3： 文档 4： 视频 5：音乐，6：其他, null:全部
     * @return
     * @description 根据目录id查询目录下的文件信息
     */
    @GetMapping("/subset")
    PageResult<StorageDto> listSubsetStorages(Pageable pageable, @RequestParam(name = "id") String id,
                                                     @RequestParam(name = "name", required = false) String name,
                                                     @RequestParam(name = "enabled", required = false) Integer enabled,
                                                     @RequestParam(name = "type", required = false) Integer type);


    /**
     * @param clientId ， 应用Id，
     * @param spaceId  ， 存储空间Id，
     * @param owner    拥有者， name 文件名称，
     * @param enabled: 0: 删除状态， 1：正常状态，
     * @param type:    0：目录 1：文件，2: 图片 3： 文档 4： 视频 5：音乐，6：其他
     * @param queryCount 1:查询份数
     * @return
     * @description 文档拥有者所有根节点文件
     */
    @GetMapping("/all/root")
    List<StorageDto> listAllRootStorages(@RequestParam(name = "clientId", required = false) String clientId,
                                         @RequestParam(name = "spaceId", required = false) String spaceId,
                                         @RequestParam(name = "owner", required = false) String owner,
                                         @RequestParam(name = "name", required = false) String name,
                                         @RequestParam(name = "enabled", required = false) Integer enabled,
                                         @RequestParam(name = "type", required = false) Integer type,
                                         @RequestParam(name = "queryCount", required = false) Integer queryCount);


    /**
     * @param id:要查询的文件节点，
     * @param name         文件名称，
     * @param enabled:     0: 删除状态， 1：正常状态
     * @param type:        0：目录 1：文件，2: 图片 3： 文档 4： 视频 5：音乐，6：其他, null:全部
     * @return
     * @description 根据目录id查询目录下的文件信息
     */
    @GetMapping("/all/subset")
    List<StorageDto> listAllSubsetStorages(@RequestParam(name = "id") String id,
                                           @RequestParam(name = "name", required = false) String name,
                                           @RequestParam(name = "enabled", required = false) Integer enabled,
                                           @RequestParam(name = "type", required = false) Integer type,
                                           @RequestParam(name = "queryCount", required = false) Integer queryCount);

    /**
     * @param  clientId ， 应用Id，
     * @param  spaceId ， 存储空间Id，
     * @param owner 为空时查询所有用户文件
     * @param name 文件名称
     * @param enabled: 0: 删除状态， 1：正常状态
     * @param type: 0：目录 1：文件，2: 图片 3： 文档 4： 视频 5：音乐，6：其他, null:全部
     * @return
     * @description 按文件名称查询所有文件
     */
    @GetMapping("/name")
    List<StorageDto> listStoragesByName(@RequestParam(name = "clientId", required = false) String clientId,
                                        @RequestParam(name = "spaceId", required = false) String spaceId,
                                        @RequestParam(name = "owner", required = false) String owner,
                                        @RequestParam(name = "name") String name,
                                        @RequestParam(name = "enabled", required = false) Integer enabled,
                                        @RequestParam(name = "type", required = false) Integer type);

    @GetMapping("/{id}/node-path")
    List<StorageDto> getNodePath(@PathVariable(name = "id") String id);


    /**
     * @param multipartDto 文件
     * @return
     * @description 上传文件
     */
    @PostMapping("/multipart/upload")
    StorageDto multipartUpload(@RequestBody MultipartDto multipartDto);

    /**
     * @param multipartDto 文件
     * @return
     * @description 替换上传文件
     */
    @PostMapping("/replace/upload")
    StorageDto replaceUpload(@RequestParam(name = "id") String id, @RequestBody MultipartDto multipartDto);


    /**
     * @param  clientId ， 应用Id，
     * @param  spaceId ， 存储空间Id，
     * @param owner 为空时查询所有用户文件
     * @param nodeId 目录Id
     * @param enabled: 0: 删除状态， 1：正常状态
     * @param type: 0：目录 1：文件，2: 图片 3： 文档 4： 视频 5：音乐，6：其他, null:全部
     * @param subpath: 是否递归所有子目录  1： 是 2：否
     * @return
     * @description 查询文件个数
     */
    @GetMapping("/count")
    long getFileCount(@RequestParam(name = "clientId", required = false) String clientId,
                      @RequestParam(name = "spaceId", required = false) String spaceId,
                      @RequestParam(name = "owner", required = false) String owner,
                      @RequestParam(name = "nodeId", required = false) String nodeId,
                      @RequestParam(name = "enabled", required = false) Integer enabled,
                      @RequestParam(name = "type", required = false) Integer type,
                      @RequestParam(name = "subpath", required = false) Integer subpath);

    /**
     * @param owner 为空时查询所有用户文件
     * @param nodeIds 目录Ids
     * @param enabled: 0: 删除状态， 1：正常状态
     * @param type: 0：目录 1：文件，2: 图片 3： 文档 4： 视频 5：音乐，6：其他, null:全部
     * @param subpath: 是否递归所有子目录  1： 是 2：否
     * @return
     * @description 批量查询文件夹下文件个数
     */
    @PostMapping("/folders/count")
    Map<String, Long> getFoldersCount(
            @RequestBody List<String> nodeIds,
            @RequestParam(name = "owner", required = false) String owner,
            @RequestParam(name = "enabled", required = false) Integer enabled,
            @RequestParam(name = "type", required = false) Integer type,
            @RequestParam(name = "subpath", required = false) Integer subpath);

    /**
     * @param ids 附件ids， 中间用','隔开
     * @param owner 压缩文件操作者
     * @param nodeId 目录Id
     * @param name: 生成压缩的文件名
     * @return
     * @description 压缩文件
     */
    @CrossOrigin
    @GetMapping("/zip")
    StorageDto zipStorages(@RequestParam(name = "ids") String ids,
                                  @RequestParam(name = "name", required = false) String name,
                                  @RequestParam(name = "owner") String owner,
                                  @RequestParam(name = "nodeId", required = false) String nodeId
    );

    /**
     * @param username 用户名
     * @return
     * @description 获取用户签名ID
     */
    @GetMapping("/user-sign")
    String userSign(@RequestParam(name = "username") String username);

    /**
     * @param nodeId 文件夹节点ID
     * @param ids 要移动的文件ID
     * @return
     * @description 移动文件
     */
    @PutMapping("/move")
    BaseResultDto moveStorages(@RequestParam(name = "nodeId") String nodeId, @RequestBody List<String> ids);

    /**
     * @param id 文件ID
     * @return BaseResultDto.code  0: 成功 msg: base64数据， 1： 失败， msg 失败信息
     * @description 获取图片base64编码数据
     */
    @GetMapping("/download/base64")
    BaseResultDto downloadBase64(@RequestParam(name = "id") String id);

    /**
     * @param id 文件ID
     * @return
     * @description 下载
     */
    @GetMapping("/download")
    MultipartDto download(@RequestParam(name = "id") String id);
}
