package cn.gtmap.gtc.storage.clients.v1;

import cn.gtmap.gtc.common.http.result.PageResult;
import cn.gtmap.gtc.storage.domain.dto.MultipartDto;
import cn.gtmap.gtc.storage.domain.dto.StorageDto;
import org.springframework.cloud.netflix.feign.FeignClient;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * @author <a href="mailto:yangyang@gtmap.cn">yangyang</a>
 * @version 1.0.2018/9/27
 * @description
 */

@FeignClient(name = "storage-app")
@RequestMapping("/storage/rest/v1/storages")
public interface StorageClient {

    /**
     * @param nodeId 上级文件夹ID,  name: 新建文件夹名， owner：拥有者
     * @return
     * @description 新建文件夹， nodeId 为空将建在用户根目录下
     */
    @PostMapping("/folder")
    StorageDto createFolder(@RequestParam(name = "spaceId", required = false) String spaceId,
                            @RequestParam(name = "proId", required = false) String proId,
                            @RequestParam(name = "nodeId", required = false) String nodeId,
                            @RequestParam(name = "name") String name,
                            @RequestParam(name = "owner") String owner);

    /**
     * @param nodeId 上级文件夹ID,  name: 新建文件夹名， owner：拥有者
     * @return
     * @description 查看是否存在
     */
    @GetMapping("/existence")
    boolean checkExist(@RequestParam(name = "spaceId", required = false) String spaceId,
                       @RequestParam(name = "proId", required = false) String proId,
                       @RequestParam(name = "nodeId", required = false) String nodeId,
                       @RequestParam(name = "name") String name,
                       @RequestParam(name = "owner", required = false) String owner,
                       @RequestParam(name = "type", required = false) Integer type);

    /**
     * @param id 文件id,  enabled: 0: 删除状态， 1：正常状态
     * @return
     * @description 更该文件状态
     */
    @PostMapping("/status")
    boolean changeStatus(@RequestParam(name = "id") String id, @RequestParam(name = "enabled") Integer enabled);


    /**
     * @param id 文件id,  name: 文件名
     * @return
     * @description 文件重命名
     */
    @PostMapping("/rename/{id}")
    boolean rename(@PathVariable(name = "id") String id, @RequestParam(name = "name") String name);

    /**
     * @param ids 文件id列表
     * @return
     * @description 删除文件
     */
    @DeleteMapping
    boolean deleteStorages(@RequestBody List<String> ids);

    /**
     * @param ids 文件
     * @return
     * @description 根据Ids查询文件信息
     */
    @PostMapping("/list")
    List<StorageDto> listStoragesByIds(@RequestBody List<String> ids);

    /**
     * @param id 文件
     * @return
     * @description 根据Id查询文件信息
     */
    @GetMapping("/{id}")
    StorageDto findById(@PathVariable(name = "id") String id);


    /**
     * @param  owner 拥有者， name 文件名称，
     * @param  enabled: 0: 删除状态， 1：正常状态，
     * @param  type: 0：目录 1：文件，2: 图片 3： 文档 4： 视频 5：音乐，6：其他
     * @return
     * @description 网盘模式-文档拥有者所有根节点文件
     */
    @GetMapping("/owner/root")
    PageResult<StorageDto> listOwnerStorages(Pageable pageable,
                                                    @RequestParam(name = "spaceId", required = false) String spaceId,
                                                    @RequestParam(name = "owner") String owner,
                                                    @RequestParam(name = "name", required = false) String name,
                                                    @RequestParam(name = "enabled", required = false) Integer enabled,
                                                    @RequestParam(name = "type", required = false) Integer type);

    /**
     * @param name 文件名称
     * @param  enabled: 0: 删除状态， 1：正常状态，
     * @param  type: 0：目录 1：文件，2: 图片 3： 文档 4： 视频 5：音乐，6：其他
     * @return
     * @description 网盘模式-管理员查看所有根节点文件
     */
    @GetMapping("/admin/root")
    PageResult<StorageDto> listAdminStorages(Pageable pageable,
                                                    @RequestParam(name = "spaceId", required = false) String spaceId,
                                                    @RequestParam(name = "name", required = false) String name,
                                                    @RequestParam(name = "enabled", required = false) Integer enabled,
                                                    @RequestParam(name = "type", required = false) Integer type);

    /**
     * @param owner 为空时查询所有用户文件
     * @param name 文件名称
     * @param enabled: 0: 删除状态， 1：正常状态
     * @param type: 0：目录 1：文件，2: 图片 3： 文档 4： 视频 5：音乐，6：其他, null:全部
     * @return
     * @description 网盘模式-按文件类型查询所有文件
     */
    @GetMapping("/mime")
    PageResult<StorageDto> listMimeStorages(Pageable pageable,
                                                        @RequestParam(name = "spaceId", required = false) String spaceId,
                                                        @RequestParam(name = "owner", required = false) String owner,
                                                        @RequestParam(name = "name", required = false) String name,
                                                        @RequestParam(name = "enabled", required = false) Integer enabled,
                                                        @RequestParam(name = "type", required = false) Integer type);

    /**
     * @param spaceId 空间id
     * @param owner 为空时查询所有用户文件
     * @param name 文件名称
     * @param enabled: 0: 删除状态， 1：正常状态
     * @param type: 0：目录 1：文件，2: 图片 3： 文档 4： 视频 5：音乐，6：其他, null:全部
     * @return
     * @description 项目模式-查询空间下文件列表
     */
    @GetMapping("/project/root")
    PageResult<StorageDto> listProjectStorages(Pageable pageable,
                                                    @RequestParam(name = "spaceId", required = false) String spaceId,
                                                    @RequestParam(name = "proId", required = false) String proId,
                                                    @RequestParam(name = "owner", required = false) String owner,
                                                    @RequestParam(name = "name", required = false) String name,
                                                    @RequestParam(name = "enabled", required = false) Integer enabled,
                                                    @RequestParam(name = "type", required = false) Integer type);

    /**
     * @param spaceId 空间id
     * @param owner 为空时查询所有用户文件
     * @param name 文件名称
     * @param enabled: 0: 删除状态， 1：正常状态
     * @param type: 0：目录 1：文件，2: 图片 3： 文档 4： 视频 5：音乐，6：其他, null:全部
     * @return
     * @description 项目模式-空间下按文件类型查询所有文件
     */
    @GetMapping("/project/mime")
    PageResult<StorageDto> listProjectMimeStorages(Pageable pageable,
                                                    @RequestParam(name = "spaceId", required = false) String spaceId,
                                                    @RequestParam(name = "proId", required = false) String proId,
                                                    @RequestParam(name = "owner", required = false) String owner,
                                                    @RequestParam(name = "name", required = false) String name,
                                                    @RequestParam(name = "enabled", required = false) Integer enabled,
                                                    @RequestParam(name = "type", required = false) Integer type);

    /**
     * @param id:要查询的文件节点， name 文件名称，
     * @param enabled: 0: 删除状态， 1：正常状态
     * @param type: 0：目录 1：文件，2: 图片 3： 文档 4： 视频 5：音乐，6：其他, null:全部
     * @return
     * @description 根据目录id查询目录下的文件信息
     */
    @GetMapping("/subset")
    PageResult<StorageDto> listSubsetStorages(Pageable pageable, @RequestParam(name = "id") String id,
                                                     @RequestParam(name = "name", required = false) String name,
                                                     @RequestParam(name = "enabled", required = false) Integer enabled,
                                                     @RequestParam(name = "type", required = false) Integer type);

    @GetMapping("/{id}/node-path")
    List<StorageDto> getNodePath(@PathVariable(name = "id") String id);


    /**
     * @param multipartDto 文件
     * @return
     * @description 上传文件
     */
    @PostMapping("/multipart/upload")
    StorageDto multipartUpload(@RequestBody MultipartDto multipartDto);

    /**
     * @param spaceId 空间id
     * @param owner 为空时查询所有用户文件
     * @param nodeId 目录Id
     * @param enabled: 0: 删除状态， 1：正常状态
     * @param type: 0：目录 1：文件，2: 图片 3： 文档 4： 视频 5：音乐，6：其他, null:全部
     * @param subpath: 是否递归所有子目录  1： 是 2：否
     * @return
     * @description 查询文件个数
     */
    @CrossOrigin
    @GetMapping("/count")
    long getFileCount(@RequestParam(name = "spaceId", required = false) String spaceId,
                             @RequestParam(name = "proId", required = false) String proId,
                             @RequestParam(name = "owner", required = false) String owner,
                             @RequestParam(name = "nodeId", required = false) String nodeId,
                             @RequestParam(name = "enabled", required = false) Integer enabled,
                             @RequestParam(name = "type", required = false) Integer type,
                             @RequestParam(name = "subpath", required = false) Integer subpath);


    /**
     * @param ids 附件ids， 中间用','隔开
     * @param owner 压缩文件操作者
     * @param nodeId 目录Id
     * @param name: 生成压缩的文件名
     * @return
     * @description 压缩文件
     */
    @CrossOrigin
    @GetMapping("/zip")
    StorageDto zipStorages(@RequestParam(name = "ids") String ids,
                                  @RequestParam(name = "name", required = false) String name,
                                  @RequestParam(name = "owner") String owner,
                                  @RequestParam(name = "nodeId", required = false) String nodeId
    );
}
