package cn.gtmap.gtc.start.config.message;

import cn.gtmap.gtc.start.config.audit.rabbit.GtcRabbitEnum;
import cn.gtmap.gtc.start.config.message.consumer.NotifyConsumer;
import cn.gtmap.gtc.start.config.message.consumer.NotifyMsgSource;
import cn.gtmap.gtc.start.config.message.producer.HttpMsgProducer;
import cn.gtmap.gtc.start.config.message.producer.MessageProducer;
import cn.gtmap.gtc.start.config.message.producer.RabbiMsgProducer;
import com.alibaba.fastjson.JSON;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.amqp.RabbitAutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.cloud.client.discovery.DiscoveryClient;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

/**
 * @author <a href="mailto:yangyang@gtmap.cn">yangyang</a>
 * @version 1.0.2020-5-14
 * @description
 */
@Configuration
public class GtcMessageAutoConfiguration {
    protected static final Logger logger = LoggerFactory.getLogger(GtcMessageAutoConfiguration.class);

    @Bean
    @ConditionalOnMissingBean(name = "messageProducer")
    public MessageProducer messageProducer(DiscoveryClient discoveryClient){
        return new HttpMsgProducer(discoveryClient);
    }

    @Configuration
    @ConditionalOnClass(name = {"org.springframework.amqp.rabbit.core.RabbitTemplate"})
    @ConditionalOnProperty(prefix = "message.sender", name = "type", havingValue = "rabbit", matchIfMissing = true)
    @Import(RabbitAutoConfiguration.class)
    protected static class RabbitMsgSenderConfiguration {
        @Bean
        public MessageProducer messageProducer(org.springframework.amqp.rabbit.core.RabbitTemplate rabbitTemplate){
            return new RabbiMsgProducer(rabbitTemplate);
        }
    }


    @Configuration
    @ConditionalOnClass(name = {"org.springframework.amqp.rabbit.core.RabbitTemplate"})
    @ConditionalOnProperty(name = "message.consumer.type", havingValue = "rabbit")
    @Import(org.springframework.boot.autoconfigure.amqp.RabbitAutoConfiguration.class)
    public static class GtcMessageConsumerConfiguration {

        private final org.springframework.amqp.rabbit.core.RabbitTemplate rabbitTemplate;
        private final NotifyConsumer notifyConsumer;
        @Value("${spring.application.name}")
        private String clientId;

        public GtcMessageConsumerConfiguration(org.springframework.amqp.rabbit.core.RabbitTemplate rabbitTemplate,NotifyConsumer notifyConsumer) {
            this.rabbitTemplate = rabbitTemplate;
            this.notifyConsumer = notifyConsumer;
        }

        private String getQueueName(){
            return GtcRabbitEnum.QueueName.FANOUT_NOTIFY_QUEUE.getName() + clientId;
        }

        @Bean({"notifyFanoutExchange"})
        org.springframework.amqp.core.FanoutExchange notifyFanoutExchange(org.springframework.amqp.rabbit.core.RabbitAdmin rabbitAdmin) {
            org.springframework.amqp.core.FanoutExchange exchange = new org.springframework.amqp.core.FanoutExchange(GtcRabbitEnum.ExchangeName.MESSAGE_FANOUT_NOTIFY.getName());
            rabbitAdmin.declareExchange(exchange);
            return exchange;
        }

        @Bean({"notifyDirectExchange"})
        org.springframework.amqp.core.DirectExchange notifyDirectExchange(org.springframework.amqp.rabbit.core.RabbitAdmin rabbitAdmin) {
            org.springframework.amqp.core.DirectExchange exchange = new org.springframework.amqp.core.DirectExchange(GtcRabbitEnum.ExchangeName.MESSAGE_DIRECT_NOTIFY.getName());
            rabbitAdmin.declareExchange(exchange);
            return exchange;
        }


        @Bean
        org.springframework.amqp.core.Queue notifyQueue(org.springframework.amqp.rabbit.core.RabbitAdmin rabbitAdmin){
            org.springframework.amqp.core.Queue queue = new org.springframework.amqp.core.Queue(getQueueName(),true, false, false, null);
            rabbitAdmin.declareQueue(queue);
            return queue;
        }

        @Bean
        org.springframework.amqp.core.Binding bindingDirectNotifyQueue(org.springframework.amqp.rabbit.core.RabbitAdmin rabbitAdmin){
            org.springframework.amqp.core.Binding  binding = org.springframework.amqp.core.BindingBuilder.bind(notifyQueue(rabbitAdmin)).to(notifyDirectExchange(rabbitAdmin)).with(clientId);
            rabbitAdmin.declareBinding(binding);
            return binding;
        }

        @Bean
        org.springframework.amqp.core.Binding bindingNotifyQueue(org.springframework.amqp.rabbit.core.RabbitAdmin rabbitAdmin){
            org.springframework.amqp.core.Binding  binding = org.springframework.amqp.core.BindingBuilder.bind(notifyQueue(rabbitAdmin)).to(notifyFanoutExchange(rabbitAdmin));
            rabbitAdmin.declareBinding(binding);
            return binding;
        }

        public org.springframework.amqp.rabbit.listener.api.ChannelAwareMessageListener handleService(){
            return new org.springframework.amqp.rabbit.listener.api.ChannelAwareMessageListener() {
                public void onMessage( org.springframework.amqp.core.Message message, com.rabbitmq.client.Channel channel) throws Exception {
                    try {
                        String body = new String(message.getBody(), "utf-8");
                        logger.debug("消息回调通知:{}", body);
                        NotifyMsgSource notify = JSON.parseObject(body, NotifyMsgSource.class);
                        notifyConsumer.recNotify(notify);
                    } catch (Exception e) {
                        logger.warn("notify.onMessagqe",e);
                    }
                }
            };
        }

        @Bean({"notifyMessageContainer"})
        public org.springframework.amqp.rabbit.listener.SimpleMessageListenerContainer notifyMessageContainer(org.springframework.amqp.rabbit.connection.ConnectionFactory connectionFactory) {
            org.springframework.amqp.rabbit.listener.SimpleMessageListenerContainer container = new org.springframework.amqp.rabbit.listener.SimpleMessageListenerContainer(connectionFactory);
            container.setQueueNames(getQueueName());
            container.setMessageListener(handleService());
            container.setAcknowledgeMode(org.springframework.amqp.core.AcknowledgeMode.AUTO);
            return container;
        }
    }
}
