package cn.gtmap.gtc.start.config.audit;

import org.springframework.boot.actuate.audit.AuditEvent;
import org.springframework.boot.actuate.security.AbstractAuthorizationAuditListener;
import org.springframework.security.access.ConfigAttribute;
import org.springframework.security.access.event.AbstractAuthorizationEvent;
import org.springframework.security.access.event.AuthenticationCredentialsNotFoundEvent;
import org.springframework.security.access.event.AuthorizationFailureEvent;
import org.springframework.security.access.event.AuthorizedEvent;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.userdetails.UserDetails;

import java.util.*;

/**
 * @author <a href="mailto:yangyang@gtmap.cn">yangyang</a>
 * @version 1.0, 2018/7/18
 * @description
 */

public class AuthorizationAuditListener extends AbstractAuthorizationAuditListener {

    public static final String AUTHORIZATION_FAILURE = "AUTHORIZATION_FAILURE";
    public static final String AUTHORIZATION_SUCCESS = "AUTHORIZATION_SUCCESS";
    private final boolean authorizationSuccess;

    public AuthorizationAuditListener(boolean authorizationSuccess) {
        this.authorizationSuccess = authorizationSuccess;
    }

    @Override
    public void onApplicationEvent(AbstractAuthorizationEvent event) {
        if (event instanceof AuthenticationCredentialsNotFoundEvent) {
            onAuthenticationCredentialsNotFoundEvent(
                    (AuthenticationCredentialsNotFoundEvent) event);
        } else if (event instanceof AuthorizationFailureEvent) {
            onAuthorizationFailureEvent((AuthorizationFailureEvent) event);
        } else if(event instanceof AuthorizedEvent && authorizationSuccess){
            onAuthorizationSuccessEvent((AuthorizedEvent) event);
        }
    }

    private void onAuthorizationSuccessEvent(AuthorizedEvent event) {
        Authentication authentication =  event.getAuthentication();
        if(authentication != null){
            String principal = "unknown";
            if(authentication.getName() != null){
                principal = authentication.getName();
            }else if(authentication.getPrincipal() instanceof UserDetails){
                principal = ((UserDetails)authentication.getPrincipal()).getUsername();
            }
            if("anonymousUser".equals(principal)){
                return;
            }

            Map<String, Object> data = new HashMap<>();
            data.put("logClassify", AuthenticationAuditListener.CLASSIFY);
            data.put("attributes",getAttributeList(event.getConfigAttributes()));
            publish(new AuditEvent(principal,AUTHORIZATION_SUCCESS, data));
        }
    }

    private void onAuthenticationCredentialsNotFoundEvent(
            AuthenticationCredentialsNotFoundEvent event) {
        Map<String, Object> data = new HashMap<>();
        data.put("exception", event.getCredentialsNotFoundException().getClass().getName());
        data.put("message", event.getCredentialsNotFoundException().getMessage());
        data.put("attributes",getAttributeList(event.getConfigAttributes()));
        data.put("logClassify", AuthenticationAuditListener.CLASSIFY);
        publish(new AuditEvent("unknown",
                AuthenticationAuditListener.AUTHENTICATION_FAILURE, data));
    }

    private List<String> getAttributeList(Collection<ConfigAttribute> attributes){
        List<String> attributeList = new ArrayList<>();
        attributes.forEach(attribute -> attributeList.add(attribute.toString().replace("'", "")));
        return attributeList;
    }

    private void onAuthorizationFailureEvent(AuthorizationFailureEvent event) {
        if("anonymousUser".equals(event.getAuthentication().getName())){
            return;
        }
        Map<String, Object> data = new HashMap<>();
        data.put("exception", event.getAccessDeniedException().getClass().getName());
        data.put("message", event.getAccessDeniedException().getMessage());
        data.put("logClassify", AuthenticationAuditListener.CLASSIFY);
        if (event.getAuthentication().getDetails() != null) {
            data.put("details", event.getAuthentication().getDetails());
        }
        data.put("attributes",getAttributeList(event.getConfigAttributes()));
        publish(new AuditEvent(event.getAuthentication().getName(), AUTHORIZATION_FAILURE,
                data));
    }
}
