package cn.gtmap.gtc.start.config.audit;

import cn.gtmap.gtc.start.config.audit.collector.DefaultLogCollector;
import cn.gtmap.gtc.start.config.audit.collector.LogCollector;
import cn.gtmap.gtc.start.config.audit.collector.SecurityLogCollector;
import cn.gtmap.gtc.start.config.audit.sender.LocalLogSender;
import cn.gtmap.gtc.start.config.audit.sender.LogSender;
import cn.gtmap.gtc.start.config.audit.sender.RabbitmqLogSender;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.actuate.audit.AuditEventRepository;
import org.springframework.boot.actuate.audit.listener.AbstractAuditListener;
import org.springframework.boot.actuate.audit.listener.AuditListener;
import org.springframework.boot.actuate.security.AbstractAuthenticationAuditListener;
import org.springframework.boot.actuate.security.AbstractAuthorizationAuditListener;
import org.springframework.boot.autoconfigure.amqp.RabbitAutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.util.StringUtils;

/**
 * @author <a href="mailto:yangyang@gtmap.cn">yangyang</a>
 * @version 1.0.2020-5-9
 * @description
 */
@Configuration
@EnableConfigurationProperties(LogClassifyProperties.class)
public class AuditAutoConfiguration {

    @Value("${spring.application.name:}")
    private String appName;

    @Value("${logging.security.collection.authorizationSuccess:false}")
    private boolean authorizationSuccess;

    @Value("${logging.security.collection.authenticationSuccess:false}")
    private boolean authenticationSuccess;

    @Bean
    @ConditionalOnClass(name = "org.springframework.security.authentication.event.AbstractAuthenticationEvent")
    @ConditionalOnMissingBean(AbstractAuthenticationAuditListener.class)
    public AuthenticationAuditListener authenticationAuditListener() throws Exception {
        return new AuthenticationAuditListener(authenticationSuccess);
    }

    @Bean
    @ConditionalOnClass(name = "org.springframework.security.access.event.AbstractAuthorizationEvent")
    @ConditionalOnMissingBean(AbstractAuthorizationAuditListener.class)
    public AuthorizationAuditListener authorizationAuditListener() throws Exception {
        return new AuthorizationAuditListener(authorizationSuccess);
    }

//    @Bean
//    @ConditionalOnMissingBean({LogSender.class})
//    @ConditionalOnProperty(prefix = "logging.store", name = "type", havingValue = "http", matchIfMissing = true)
//    public LogSender logSender(DiscoveryClient discoveryClient){
//        return new HttpLogSender(discoveryClient);
//    }

    @Configuration
    @ConditionalOnClass(name = {"org.springframework.amqp.rabbit.core.RabbitTemplate"})
    @ConditionalOnProperty(prefix = "logging.store", name = "type", havingValue = "rabbit", matchIfMissing = true)
    @Import(RabbitAutoConfiguration.class)
    protected static class RabbitLogConfiguration {
        @Bean
        public LogSender logSender(org.springframework.amqp.rabbit.core.RabbitTemplate rabbitTemplate){
            return new RabbitmqLogSender(rabbitTemplate);
        }
    }

    @Configuration
    @ConditionalOnClass(name = {"org.springframework.amqp.rabbit.core.RabbitTemplate", "zipkin2.reporter.Sender"})
    @ConditionalOnProperty(prefix = "spring.zipkin.sender", name = "type", havingValue = "rabbit", matchIfMissing = true)
    @Import(RabbitAutoConfiguration.class)
    protected static class RabbitZipkinConfiguration {
        @Bean
        public zipkin2.reporter.Sender zipkinSender2(org.springframework.boot.autoconfigure.amqp.RabbitProperties rabbitProperties) {
            com.rabbitmq.client.ConnectionFactory connectionFactory = new com.rabbitmq.client.ConnectionFactory();
            connectionFactory.setHost(rabbitProperties.getHost());
            connectionFactory.setPort(rabbitProperties.getPort());
            connectionFactory.setPassword(rabbitProperties.getPassword());
            connectionFactory.setUsername(rabbitProperties.getUsername());
            String addr = rabbitProperties.getAddresses();
            if (StringUtils.isEmpty(rabbitProperties.getAddresses())){
                addr = rabbitProperties.getHost()+":"+rabbitProperties.getPort();
            }
            return zipkin2.reporter.amqp.RabbitMQSender.newBuilder()
                    .connectionFactory(connectionFactory)
                    .queue("zipkin")
                    .addresses(addr)
                    .build();
        }
    }


    @Bean
    @ConditionalOnMissingBean({LogSender.class})
    public LogSender logSender(){
        return new LocalLogSender();
    }


    @Configuration
    @ConditionalOnClass(name = "org.springframework.security.core.context.SecurityContextHolder")
    protected static class SecurityLogCollectorCfg {
        @Bean
        public LogCollector logCollector() {
            return new SecurityLogCollector();
        }
    }

    @Bean
    @ConditionalOnMissingBean
    public LogCollector logCollector(){
        return new DefaultLogCollector();
    }

    @Bean
    @ConditionalOnMissingBean(AbstractAuditListener.class)
    public AuditListener auditListener(LogClassifyProperties logClassifyProperties, LogSender logSender, LogCollector logCollector) throws Exception {
        return new AuditListener(zipkinAuditEventRepository(logClassifyProperties,logSender, logCollector));
    }

    @Bean
    @ConditionalOnMissingBean(AuditEventRepository.class)
    public ZipkinAuditEventRepository zipkinAuditEventRepository(LogClassifyProperties logClassifyProperties, LogSender logSender, LogCollector logCollector) {
        return new ZipkinAuditEventRepository(appName,logClassifyProperties,logSender, logCollector);
    }

    @Bean
    public CustomAuditEventRepository customAuditEventRepository(LogClassifyProperties logClassifyProperties, LogSender logSender, LogCollector logCollector) {
        return new CustomAuditEventRepository(logSender, logCollector);
    }

}
