package cn.gtmap.gtc.gis.utils;

import org.apache.commons.collections.CollectionUtils;
import org.apache.tools.zip.ZipEntry;
import org.apache.tools.zip.ZipFile;
import org.apache.tools.zip.ZipOutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;

import static cn.gtmap.gtc.gis.Constant.CHARSET_UTF8;
import static cn.gtmap.gtc.gis.Constant.SUFFIX_ZIP;

/**
 * Created by Fjj on 2017/11/16.
 * 压缩或解压zip：
 * 由于直接使用java.util.zip工具包下的类，会出现中文乱码问题，所以使用ant.jar中的org.apache.tools.zip下的工具类
 */
public class ZipUtil {

    private ZipUtil() {

    }

    private static final Logger logger = LoggerFactory.getLogger(ZipUtil.class.getClass());

    /**
     * 压缩文件或路径
     *
     * @param zip      压缩的目的地址
     * @param srcFiles 压缩的源文件
     */
    public static void zipFile(String zip, List<File> srcFiles) {
        if (CollectionUtils.isNotEmpty(srcFiles)) {
            zipFile(zip, srcFiles.toArray(new File[0]));
        }
    }

    /**
     * 压缩文件或路径
     *
     * @param zip      压缩的目的地址
     * @param srcFiles 压缩的源文件
     */
    public static void zipFile(String zip, File[] srcFiles) {
        try {
            if (zip.endsWith(SUFFIX_ZIP) || zip.endsWith(SUFFIX_ZIP.toUpperCase())) {
                ZipOutputStream zipOut = new ZipOutputStream(new FileOutputStream(new File(zip)));
                zipOut.setEncoding(CHARSET_UTF8);
                for (File _f : srcFiles) {
                    handlerFile(zip, zipOut, _f, "");
                }
                zipOut.close();
            }
        } catch (FileNotFoundException e) {
            logger.error("压缩zip文件时未找到指定文件", e);
        } catch (IOException e) {
            logger.error("压缩zip文件时发生错误", e);
        }
    }

    /**
     * @param zip     压缩的目的地址
     * @param zipOut
     * @param srcFile 被压缩的文件信息
     * @param path    在zip中的相对路径
     * @throws IOException
     */
    private static void handlerFile(String zip, ZipOutputStream zipOut, File srcFile, String path) throws IOException {
        if (!"".equals(path) && !path.endsWith(File.separator)) {
            path += File.separator;
        }
        if (!srcFile.getPath().equals(zip)) {
            if (srcFile.isDirectory()) {
                File[] files = srcFile.listFiles();
                if (files.length == 0) {
                    zipOut.putNextEntry(new ZipEntry(path + srcFile.getName() + File.separator));
                    zipOut.closeEntry();
                } else {
                    for (File file : files) {
                        handlerFile(zip, zipOut, file, path + srcFile.getName());
                    }
                }
            } else {
                try (InputStream inputStream = new FileInputStream(srcFile);) {
                    zipOut.putNextEntry(new ZipEntry(path + srcFile.getName()));
                    int len = 0;
                    byte[] bytes = new byte[1024];
                    while ((len = inputStream.read(bytes)) > 0) {
                        zipOut.write(bytes, 0, len);
                    }
                } catch (IOException e) {
                    logger.error(e.toString());
                } finally {
                    zipOut.closeEntry();
                }
            }
        }
    }

    /**
     * 压缩文件
     *
     * @param zip          压缩的目的地址
     * @param files        压缩的源文件
     * @param fileNameList 文件名列表
     */
    public static void zipFile(String zip, List<File> files, List<String> fileNameList) {
        try {
            if (zip.endsWith(SUFFIX_ZIP) || zip.endsWith(SUFFIX_ZIP.toUpperCase())) {
                ZipOutputStream zipOut = new ZipOutputStream(new FileOutputStream(new File(zip)));
                zipOut.setEncoding(CHARSET_UTF8);
                for (int i = 0; i < files.size(); i++) {
                    handlerFile(zipOut, files.get(i), "", fileNameList.get(i));
                }
                zipOut.close();
            }
        } catch (FileNotFoundException e) {
            logger.error("压缩zip文件时未找到指定文件", e);
        } catch (IOException e) {
            logger.error("压缩zip文件时发生错误", e);
        }
    }

    /**
     * @param zipOut  Zip输出数据流
     * @param srcFile 被压缩的文件
     * @param path    在zip中的相对路径
     * @throws IOException
     */
    private static void handlerFile(ZipOutputStream zipOut, File srcFile, String path, String fileName) throws IOException {
        if (!"".equals(path) && !path.endsWith(File.separator)) {
            path += File.separator;
        }
        zipOut.putNextEntry(new ZipEntry(path + fileName));
        int len = 0;
        byte[] bytes = new byte[1024];
        try (FileInputStream inputStream = new FileInputStream(srcFile)) {
            while ((len = inputStream.read(bytes)) > 0) {
                zipOut.write(bytes, 0, len);
            }
        } catch (IOException e) {
            logger.error(e.toString());
        } finally {
            zipOut.closeEntry();
        }

    }

    /**
     * 解压缩ZIP文件，将ZIP文件里的内容解压到targetDIR目录下
     *
     * @param zipPath 待解压缩的ZIP文件名
     * @param descDir 目标目录
     */
    public static List<File> upzipFile(String zipPath, String descDir) {
        return upzipFile(new File(zipPath), descDir);
    }

    /**
     * 对.zip文件进行解压缩
     *
     * @param zipFile 解压缩文件
     * @param descDir 压缩的目标地址，如：D:\\测试 或 /mnt/d/测试
     * @return
     */
    @SuppressWarnings("rawtypes")
    public static List<File> upzipFile(File zipFile, String descDir) {
        List<File> list = new ArrayList<>();
        try (ZipFile newZipFile = new ZipFile(zipFile, CHARSET_UTF8);) {
            for (Enumeration entries = newZipFile.getEntries(); entries.hasMoreElements(); ) {
                ZipEntry entry = (ZipEntry) entries.nextElement();
                File file = new File(descDir + File.separator + entry.getName());
                if (entry.isDirectory()) {
                    file.mkdirs();
                } else {
                    File parent = file.getParentFile();
                    if (!parent.exists()) {
                        parent.mkdirs();
                    }

                    try (InputStream inputStream = newZipFile.getInputStream(entry);
                         OutputStream outputStream = new FileOutputStream(file);) {
                        byte[] bytes = new byte[1024];
                        int len = 0;
                        while ((len = inputStream.read(bytes)) > 0) {
                            outputStream.write(bytes, 0, len);
                        }
                    } catch (IOException e) {
                        logger.error(e.toString());
                    }
                    list.add(file);
                }
            }
        } catch (IOException e) {
            logger.error("解压缩zip文件时发生错误", e);
        }
        return list;
    }

    /**
     * 压缩文件(.zip)的函数
     *
     * @param zipDirectory:(需要)压缩的文件夹路径
     * @param zipPath:文件压缩后放置的路径,该路径可以为null,null表示压缩到原文件的同级目录
     * @return :返回一个压缩好的文件(File),否则返回null
     */
    public static File doZip(String zipDirectory, String zipPath) {
        File zipDir = new File(zipDirectory);

        if (zipPath == null) {
            zipPath = zipDir.getParent();
        }

        // 压缩后生成的zip文件名
        String zipFileName = zipPath + File.separator + zipDir.getName() + ".zip";

        try {
            ZipOutputStream zipOut = new ZipOutputStream(new BufferedOutputStream(
                    new FileOutputStream(zipFileName)));

            // 压缩文件
            handleDir(zipDir, zipDir.getParent().length() + 1, zipOut);

            zipOut.close();
            return new File(zipFileName);
        } catch (IOException e) {
            logger.error(e.toString());
            throw new RuntimeException(e.getLocalizedMessage());
        }
    }

    /**
     * 由doZip调用,递归完成目录文件读取
     *
     * @param dir:(需要)压缩的文件夹(File           类型)
     * @param len:一个参数(记录压缩文件夹的parent路径的长度)
     * @param zipOut:需要压缩进的压缩文件
     * @throws IOException :如果出错,会抛出IOE异常
     */
    private static void handleDir(File dir, int len, ZipOutputStream zipOut)
            throws IOException {
        File[] files = dir.listFiles();
        int readedBytes;
        byte[] buf;
        if (files != null) {
            if (files.length > 0) {
                for (File fileName : files) {
                    if (fileName.isDirectory()) {
                        handleDir(fileName, len, zipOut);
                    } else {
                        try (FileInputStream fileIn = new FileInputStream(fileName);) {
                            buf = new byte[fileIn.available()];
                            zipOut.putNextEntry(new ZipEntry(fileName.getPath()
                                    .substring(len + dir.getName().length()).replaceAll("\\\\", "")));
                            while ((readedBytes = fileIn.read(buf)) > 0) {
                                zipOut.write(buf, 0, readedBytes);
                            }
                        } catch (IOException e) {
                            logger.error(e.toString());
                        } finally {
                            zipOut.closeEntry();
                        }
                    }
                }
            } else { // 如果目录为空,则单独创建之.
                zipOut.putNextEntry(new ZipEntry(dir.getPath().substring(len)
                        + "/"));
                zipOut.closeEntry();
            }
        } else {// 如果是一个单独的文件
            try (FileInputStream fileIn = new FileInputStream(dir);) {
                buf = new byte[fileIn.available()];
                zipOut.putNextEntry(new ZipEntry(dir.getPath().substring(len)));
                while ((readedBytes = fileIn.read(buf)) > 0) {
                    zipOut.write(buf, 0, readedBytes);
                }
            } catch (IOException e) {
                logger.error(e.toString());
            } finally {
                zipOut.closeEntry();
            }
        }
    }

}