package main

import (
	"bytes"
	"encoding/json"
	"fmt"
	"strconv"
	"time"
    "sync/atomic"

	"github.com/hyperledger/fabric/core/chaincode/shim"
	"github.com/hyperledger/fabric/protos/peer"
)

// 基础结构体
type BaseChaincode struct {

}

<#if (beanDefinitionList?? && beanDefinitionList?size>0)>
	<#list beanDefinitionList as beanDefinition>

type ${beanDefinition.beanName} struct {<#if beanDefinition.beanDesc??>// ${beanDefinition.beanDesc}</#if>

    <#if (beanDefinition.goFieldDefinitionList?? && beanDefinition.goFieldDefinitionList?size>0)>
        <#list beanDefinition.goFieldDefinitionList as fieldDefinition>
    ${fieldDefinition.fieldName} ${fieldDefinition.fieldTypeExpr!}  `json:${fieldDefinition.fieldName}`     // ${fieldDefinition.fieldDesc!}
        </#list>
    </#if>
}
	</#list>
</#if>


// 初始化方法，在实例化智能合约时需要调用，可不做任何事，只保留最后一行
func (t *BaseChaincode) Init(stub shim.ChaincodeStubInterface) peer.Response {

    return shim.Success(nil)
}

// invoke 方法需要将其他“其他方法名（调用时候的传参）”限定
func (t *BaseChaincode) Invoke(stub shim.ChaincodeStubInterface) peer.Response {

    fn, args := stub.GetFunctionAndParameters()

	// 设定或变更值
	if fn == "getHistory" {
		// 获取历史记录
		return t.getHistory(stub, args)
	} else if fn == "save" {
		// 保存数据
		return t.save(stub, args)
	} else if fn == "findById" {
		// 根据id获取
		return t.findById(stub, args)
	} else if fn == "deleteById" {
		// 根据id删除
		return t.deleteById(stub, args)
	} else if fn == "richQuery" {
		// 富查询
		return t.richQuery(stub, args)
	} else if fn == "set" {
		// 覆盖设置
		return t.set(stub, args)
	} else if fn == "richQueryWithPagination" {
        // 富查询带分页
		return t.richQueryWithPagination(stub, args)
    }

	return shim.Error("Invoke fn error")
}

// 获取历史记录
func (t *BaseChaincode) getHistory(stub shim.ChaincodeStubInterface, args []string) peer.Response {
	if len(args) != 1 {
		return shim.Error("Incorrect number of arguments. Expecting 1")
	}
	it, err := stub.GetHistoryForKey(args[0])
	if err != nil {
		return shim.Error(err.Error())
	}
	var result, _ = getHistoryListResult(it)
	return shim.Success(result)
}

func getHistoryListResult(resultsIterator shim.HistoryQueryIteratorInterface) ([]byte, error) {

	defer resultsIterator.Close()
	// buffer is a JSON array containing QueryRecords
	var buffer bytes.Buffer
	buffer.WriteString("[")

	bArrayMemberAlreadyWritten := false
	for resultsIterator.HasNext() {
		response, err := resultsIterator.Next()
		if err != nil {
			return nil, err
		}
		// Add a comma before array members, suppress it for the first array member
		if bArrayMemberAlreadyWritten == true {
			buffer.WriteString(",")
		}
		buffer.WriteString("{\"TxId\":")
		buffer.WriteString("\"")
		buffer.WriteString(response.TxId)
		buffer.WriteString("\"")
		buffer.WriteString(", \"Value\":")
		if response.IsDelete {
			buffer.WriteString("null")
		} else {
			buffer.WriteString(string(response.Value))
		}
		buffer.WriteString(", \"Timestamp\":")
		buffer.WriteString("\"")
		buffer.WriteString(time.Unix(response.Timestamp.Seconds, int64(response.Timestamp.Nanos)).String())
		buffer.WriteString("\"")

		buffer.WriteString(", \"IsDelete\":")
		buffer.WriteString("\"")
		buffer.WriteString(strconv.FormatBool(response.IsDelete))
		buffer.WriteString("\"")

		buffer.WriteString("}")

		bArrayMemberAlreadyWritten = true
	}
	buffer.WriteString("]")
	fmt.Printf("queryResult:\n%s\n", buffer.String())
	return buffer.Bytes(), nil
}

// 新增对象，根据 key 值
func (t *BaseChaincode) save(stub shim.ChaincodeStubInterface, args []string) peer.Response {
	daJson := ${primaryBeanName}{}

    if len(args) != 1 {
		return shim.Error("Incorrect number of arguments. Expecting 1")
	}

    var data []byte = []byte(args[0])

    var err error

	if bytes.Count(data,nil) > 1 {
		err = json.Unmarshal(data, &daJson)
	} else {
	    return shim.Error(err.Error())
	}


	if len(daJson.${primaryKey}) == 0{
		out := TimeUUID()
		daJson.${primaryKey} = out.String()
	}

	jsons, err := json.Marshal(daJson) //转换成JSON返回的是byte[]
	if err != nil {
		return shim.Error(err.Error())
	}

    err = stub.PutState(daJson.${primaryKey}, jsons)
	if err != nil {
		return shim.Error(err.Error())
	}

    return shim.Success(nil)
}

// 获取对象，根据 key 值
func (t *BaseChaincode) findById(stub shim.ChaincodeStubInterface, args []string) peer.Response {

	value, err := stub.GetState(args[0])

	if err != nil {
		return shim.Error(err.Error())
	}

	return shim.Success(value)
}

// 删除对象，根据 key 值
func (t *BaseChaincode) deleteById(stub shim.ChaincodeStubInterface, args []string) peer.Response {
	if len(args) != 1 {
		return shim.Error("Incorrect number of arguments. Expecting 1")
	}

	err := stub.DelState(args[0])

	if err != nil {
		return shim.Error("Failed to delete state")
	}

	return shim.Success(nil)
}

// 富查询：必须选用CouchDB才行
func (t *BaseChaincode) richQuery(stub shim.ChaincodeStubInterface, args []string) peer.Response {

    var buffer bytes.Buffer

	if len(args) == 0 {
		return shim.Error("Incorrect number of arguments. Expecting more than 0")
	} else {


        buffer.WriteString(args[0])
    }

	resultsIterator, err := stub.GetQueryResult(buffer.String())
	if err != nil {
		return shim.Error(err.Error())
	}

	result, err := getListResult(resultsIterator)
	if err != nil {
		return shim.Error(err.Error())
	}
	return shim.Success(result)
}

func getListResult(resultsIterator shim.StateQueryIteratorInterface) ([]byte, error) {

	defer resultsIterator.Close()
	// buffer is a JSON array containing QueryRecords
	var buffer bytes.Buffer
	buffer.WriteString("[")

	bArrayMemberAlreadyWritten := false

	for resultsIterator.HasNext() {
		queryResponse, err := resultsIterator.Next()
		if err != nil {
			return nil, err
		}
		// Add a comma before array members, suppress it for the first array member
		if bArrayMemberAlreadyWritten == true {
			buffer.WriteString(",")
		}
		buffer.WriteString("{\"Key\":")
		buffer.WriteString("\"")
		buffer.WriteString(queryResponse.Key)
		buffer.WriteString("\"")

		buffer.WriteString(", \"Record\":")
		// Record is a JSON object, so we write as-is
		buffer.WriteString(string(queryResponse.Value))
		buffer.WriteString("}")
		bArrayMemberAlreadyWritten = true
	}
	buffer.WriteString("]")
	fmt.Printf("queryResult:\n%s\n", buffer.String())
	return buffer.Bytes(), nil

}

// 富查询（带分页）：必须选用CouchDB才行
func (t *BaseChaincode) richQueryWithPagination(stub shim.ChaincodeStubInterface, args []string) peer.Response {

    var buffer bytes.Buffer

	if len(args) != 3 {
		return shim.Error("Incorrect number of arguments. Expecting 3")
	} else {
        buffer.WriteString(args[0])
    }

    num, err := strconv.ParseInt(args[1], 10, 32)

    resultsIterator, responseMetadata, err := stub.GetQueryResultWithPagination(buffer.String(), int32(num), args[2])

	if err != nil {
		return shim.Error(err.Error())
	}
	defer resultsIterator.Close()

	var resultBuffer bytes.Buffer
	resultBuffer.WriteString("{\"Records\":[")

	bArrayMemberAlreadyWritten := false
	for resultsIterator.HasNext() {
		queryResponse, err := resultsIterator.Next()
		if err != nil {
			return shim.Error(err.Error())
		}

		if bArrayMemberAlreadyWritten == true {
			resultBuffer.WriteString(",")
		}
		resultBuffer.WriteString("{\"Key\":")
		resultBuffer.WriteString("\"")
		resultBuffer.WriteString(queryResponse.Key)
		resultBuffer.WriteString("\"")

		resultBuffer.WriteString(", \"Record\":")

		resultBuffer.WriteString(string(queryResponse.Value))
		resultBuffer.WriteString("}")
		bArrayMemberAlreadyWritten = true
	}
	resultBuffer.WriteString("]")

	if err != nil {
		return shim.Error(err.Error())
	}
	resultBuffer.WriteString(", \"Bookmark\":")
	resultBuffer.WriteString("\"")
	resultBuffer.WriteString(responseMetadata.Bookmark)
	resultBuffer.WriteString("\"}")

	return shim.Success(resultBuffer.Bytes())
}

// 覆盖对象，根据 key 值
func (t *BaseChaincode) set(stub shim.ChaincodeStubInterface, args []string) peer.Response {
	if len(args) != 2 {
		return shim.Error("Incorrect number of arguments. Expecting 2")
	}

	daJson := ${primaryBeanName}{}

    var err error

	value, err := stub.GetState(args[0])

	if err != nil {
		return shim.Error(err.Error())
	}
    if bytes.Count(value,nil) < 1 {
        return shim.Error("Unexist Data.")
	}

    var data []byte = []byte(args[1])

	if bytes.Count(data,nil) > 1 {
		err = json.Unmarshal(data, &daJson)
	} else {
	    return shim.Error(err.Error())
	}

    daJson.Jlsj = time.Now().Unix()

	jsons, err := json.Marshal(daJson) //转换成JSON返回的是byte[]
	if err != nil {
		return shim.Error(err.Error())
	}

    err = stub.PutState(args[0], jsons)
	if err != nil {
		return shim.Error(err.Error())
	}

    return shim.Success(nil)
}

// UUID 生成
//out := TimeUUID()
//out.String()
type UUID [16]byte

var timeBase = time.Date(1582, time.October, 15, 0, 0, 0, 0, time.UTC).Unix()
var hardwareAddr []byte
var clockSeq uint32

func TimeUUID() UUID {
    return FromTime(time.Now())
}

func FromTime(aTime time.Time) UUID {
    var u UUID

    utcTime := aTime.In(time.UTC)
    t := uint64(utcTime.Unix()-timeBase)*10000000 + uint64(utcTime.Nanosecond()/100)
    u[0], u[1], u[2], u[3] = byte(t>>24), byte(t>>16), byte(t>>8), byte(t)
    u[4], u[5] = byte(t>>40), byte(t>>32)
    u[6], u[7] = byte(t>>56)&0x0F, byte(t>>48)

    clock := atomic.AddUint32(&clockSeq, 1)
    u[8] = byte(clock >> 8)
    u[9] = byte(clock)

    copy(u[10:], hardwareAddr)

    u[6] |= 0x10 // set version to 1 (time based uuid)
    u[8] &= 0x3F // clear variant
    u[8] |= 0x80 // set to IETF variant

    return u
}

func (u UUID) String() string {
    var offsets = [...]int{0, 2, 4, 6, 9, 11, 14, 16, 19, 21, 24, 26, 28, 30, 32, 34}
    const hexString = "0123456789abcdef"
    r := make([]byte, 36)
    for i, b := range u {
        r[offsets[i]] = hexString[b>>4]
        r[offsets[i]+1] = hexString[b&0xF]
    }
    r[8] = '-'
    r[13] = '-'
    r[18] = '-'
    r[23] = '-'
    return string(r)
}


// 实例化智能合约时需要使用
func main() {
    if err := shim.Start(new(BaseChaincode)); err != nil {
        fmt.Printf("Error starting BaseChaincode chaincode:%s", err)
    }
}