package cn.gtmap.estateplat.core.support.spring;

import cn.gtmap.estateplat.core.ex.AppException;
import cn.gtmap.estateplat.core.ex.Sourceable;
import cn.gtmap.estateplat.utils.ExUtils;
import com.google.common.collect.Lists;
import com.gtis.config.AppConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.jdbc.datasource.LazyConnectionDataSourceProxy;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.List;

/**
 * 
 *
 * @author <a href="mailto:yangxin@gtmap.cn">yangxin</a>
 * @version V1.0, 14-7-2
 */
public class DataSourceProxy extends LazyConnectionDataSourceProxy implements FactoryBean<DataSource> {
    private static final Logger LOG = LoggerFactory.getLogger(DataSourceProxy.class);
    private boolean testConnect = true;

    public void setTestConnect(boolean testConnect) {
        this.testConnect = testConnect;
    }

    @Override
    public DataSource getObject() throws Exception {
        return this;
    }

    @Override
    public Class<?> getObjectType() {
        return DataSource.class;
    }

    @Override
    public boolean isSingleton() {
        return true;
    }

    @Override
    public void afterPropertiesSet() {
        LOG.info("Db connect to [{}] with user {}", getProp("url"), getProp("username"));
        super.afterPropertiesSet();
        if (testConnect) {
            try {
                testConnect(this);
            } catch (SQLException e) {
                List<String> list = Lists.newArrayList();
                list.add("Driver -> " + getProp("driver"));
                list.add("Url -> " + getProp("url"));
                list.add("User -> " + getProp("username"));
                throw new ConnectException(e, list);
            }
        }
    }

    private String getProp(String name) {
        return AppConfig.getPlaceholderValue("db." + name);
    }

    private void testConnect(DataSource dataSource) throws SQLException {
        String driver = getProp("driver");
        String testSql = null;
        if ("com.mysql.jdbc.Driver".equals(driver)) {
            testSql = "SELECT 1";
        } else if ("org.postgresql.Driver".equals(driver)) {
            testSql = "SELECT NOW()";
        } else if ("com.ibm.db2.jdbc.app.DB2Driver".equals(driver)) {
            testSql = "SELECT CURRENT SQLID FROM SYSIBM.SYSDUMMY1";
        } else if ("oracle.jdbc.OracleDriver".equals(driver)) {
            testSql = "SELECT SYSDATE FROM DUAL";
        }
        Connection connection = null;
        try {
            connection = dataSource.getConnection();
            if (testSql != null) {
                Statement ps = null;
                try {
                    ps = connection.createStatement();
                    ps.execute(testSql);
                } finally {
                    if (ps != null) {
                        ps.close();
                    }
                }
            } else {
                connection.getMetaData().getTables(null, null, "DSPING", new String[]{"TABLE"});
            }
        } finally {
            if (connection != null) {
                connection.close();
            }
        }
    }

    private class ConnectException extends AppException implements Sourceable {
        private static final long serialVersionUID = -6626317176869586154L;
        private List<String> source;

        private ConnectException(Throwable cause, List<String> source) {
            super(null, cause);
            this.source = source;
        }

        @Override
        public String getTitle() {
            return "Database connect error";
        }

        @Override
        public String getDescription() {
            return ExUtils.buildNestedMessage(getCause());
        }

        @Override
        public String getFile() {
            return "dbConfig";
        }

        @Override
        public List<String> getLines() {
            return source;
        }

        @Override
        public int getLineNumber() {
            return 2;
        }
    }
}
