package cn.gtmap.estateplat.log;

import cn.gtmap.estateplat.core.ex.AppException;
import cn.gtmap.estateplat.model.server.core.BdcXtLog;
import com.alibaba.fastjson.JSON;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.gtis.common.util.UUIDGenerator;
import com.gtis.web.SessionUtil;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.After;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.annotation.Pointcut;
import org.springframework.data.domain.Pageable;
import org.springframework.ui.Model;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.lang.reflect.Method;
import java.util.Calendar;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * 审计日志Aspect
 * @author <a href="mailto:shenjian@gtmap.cn">jane</a>
 * @version 1.0, 2015/12/6
 */
@Aspect
public class AuditLogAspect {
    private AuditLogService auditLogService;
    private ExecutorService executor = Executors.newSingleThreadExecutor();
    private final static String ANONYMOUS_USER_ID="-999";
    private final static String ANONYMOUS_USER_NAME="匿名用户";

    public void setAuditLogService(AuditLogService auditLogService) {
        this.auditLogService = auditLogService;
    }

    @Pointcut("@annotation(cn.gtmap.estateplat.log.AuditLog)")
    public void auditLogServiceAspect(){}

    @Before("auditLogServiceAspect()")
    public void doBefore(JoinPoint joinPoint){
        AuditLog auditMethodLog = getMethodLog(joinPoint);
        auditLog(auditMethodLog.name(),getAuditContent(joinPoint,auditMethodLog));
    }

    @After("auditLogServiceAspect()")
    public void doAfter(JoinPoint joinPoint){

    }

    private void auditLog(String name,String content){
        final BdcXtLog bdcXtLog = new BdcXtLog();
        bdcXtLog.setLogid(UUIDGenerator.generate());
        bdcXtLog.setCzrq(Calendar.getInstance().getTime());
        if(StringUtils.isNotBlank(SessionUtil.getCurrentUserId())) {
            bdcXtLog.setUserid(SessionUtil.getCurrentUserId());
            bdcXtLog.setUsername(SessionUtil.getCurrentUser().getUsername());
        }else{
            bdcXtLog.setUserid(ANONYMOUS_USER_ID);
            bdcXtLog.setUsername(ANONYMOUS_USER_NAME);
        }
        bdcXtLog.setController(name);
        bdcXtLog.setParmjson(content);
        executor.submit(new Runnable() {
            @Override
            public void run() {
                auditLogService.saveAuditLog(bdcXtLog);
            }
        });
    }

    private AuditLog getMethodLog(JoinPoint joinPoint){
        try {
            String targetName = joinPoint.getTarget().getClass().getName();
            String methodName = joinPoint.getSignature().getName();
            Object[] arguments = joinPoint.getArgs();
            Class targetClass = Class.forName(targetName);
            Method[] methods = targetClass.getMethods();
            for (Method method : methods) {
                if (method.getName().equals(methodName)) {
                    Class[] clazzs = method.getParameterTypes();
                    if (clazzs.length == arguments.length) {
                        return method.getAnnotation(AuditLog.class);
                    }
                }
            }
        } catch (ClassNotFoundException e) {
            throw new AppException(e.getMessage());
        }

        return null;
    }

    private String getAuditContent(JoinPoint joinPoint,AuditLog auditLog){
        Map contentMap = Maps.newHashMap();
        String description = null;
        if(StringUtils.isNotBlank(auditLog.description())){
            description = auditLog.description();
        }else
            description = "操作内容";
        contentMap.put(description,filterArguments(joinPoint.getArgs()));
        return JSON.toJSONString(contentMap);
    }

    /**
     * 过滤掉一些无关的参数
     * @param args
     * @return
     */
    private Object filterArguments(Object[] args){
        List<Object> arguments = Lists.newArrayList();
        for(Object arg: args){
            if(arg instanceof Model || arg instanceof Pageable || arg instanceof RedirectAttributes || arg instanceof HttpServletResponse
                    || arg instanceof HttpServletRequest)
                continue;
            arguments.add(arg);
        }
        return arguments;
    }
}
