/*
 * Project:  hydroplat-parent
 * Module:   hydroplat-common
 * File:     UserService.java
 * Modifier: yangxin
 * Modified: 2014-06-12 21:35
 *
 * Copyright (c) 2014 Mapjs All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent
 * or the registration of a utility model, design or code.
 */

package cn.gtmap.egovplat.service;

import cn.gtmap.egovplat.core.data.Pageable;
import cn.gtmap.egovplat.security.ex.UserLockedException;
import cn.gtmap.egovplat.core.data.Page;
import cn.gtmap.egovplat.model.identity.Department;
import cn.gtmap.egovplat.model.identity.User;
import cn.gtmap.egovplat.security.ex.PasswordException;
import cn.gtmap.egovplat.security.ex.UserNotFoundException;
import com.mysema.query.types.Predicate;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 用户管理服务.
 * <p/>
 *
 * @author <a href="mailto:yangxin@gtmap.cn">yangxin</a>
 * @version V1.0, 14-6-12
 */
public interface UserService {

    /**
     * 根据id获取用户
     *
     * @param id 主键
     * @return 用户
     * @throws UserNotFoundException 用户未找到
     */
    User getUser(String id) throws UserNotFoundException;

    /**
     * 根据id批量获取用户
     *
     * @param ids id列表
     * @return 用户map, key为用户id, value为用户
     */
    Map<String, User> mgetUsers(Collection<String> ids);

    /**
     * 根据用户名获取用户
     *
     * @param name 用户名
     * @return 用户
     */
    User getUserByName(String name);

    /**
     * 校验登录密码
     *
     * @param name     用户名
     * @param password 密码
     * @throws UserNotFoundException 用户未找到
     * @throws cn.gtmap.egovplat.security.ex.UserLockedException   用户已被锁定
     * @throws PasswordException     密码错误
     */
    User validatePassword(String name, String password) throws UserNotFoundException, UserLockedException, PasswordException;

    /**
     * 查找用户
     *
     * @param predicate 组合条件
     * @param request   分页参数
     * @return 用户分页对象
     */
    Page<User> findUser(Predicate predicate, Pageable request);

    /**
     * 更新密码
     *
     * @param id       用户id
     * @param password 密码
     */
    void updatePassword(String id, String password);

    /**
     * 保存用户
     *
     * @param user 用户
     * @return 用户
     */
    User saveUser(User user);
    User mergeUser(User user);

    /**
     * 保存多个用户
     *
     * @param users 用户列表
     */
    void saveUsers(Collection<User> users);

    /**
     * 删除多个用户
     *
     * @param ids 用户id列表
     */
    void removeUser(Collection<String> ids);

    /**
     * 删除单个用户
     * @param id  用户id
     */
    void deleteUser(String id);

    /**
     * 根据查询条件获取记录数
     * li ning
     * @param predicates
     * @return
     */
    int getCount(Predicate... predicates);

    /**
     * 根据部门名分页查询用户
     * li ning
     * @param departmentName
     * @param pageIndex 第几页，从1开始
     * @param pageSize 每页记录数
     * @return
     */
    public List<User> searchUserByDepartmentName(String departmentName,int pageIndex,int pageSize);

    /**
     * 根据部门名查询用户个数
     * li ning
     * @param departmentName
     * @return
     */
    public int getUserCountByDepartmentName(String departmentName);

    /**
     * 根据角色名分页查询用户
     * li ning
     * @param roleName
     * @param pageIndex
     * @param pageSize
     * @return
     */
    public List<User> searchUserByRoleName(String roleName,int pageIndex,int pageSize);

    /**
     * 根据角色名分页查询用户总记录数
     * li ning
     * @param roleName
     * @return
     */
    public int getUserCountByRoleName(String roleName);

    /**
     * 根据用户名分页查询用户
     * li ning
     * @param userName
     * @param pageIndex
     * @param pageSize
     * @return
     */
    public List<User> searchUserByUserName(String userName,int pageIndex,int pageSize);

    /**
     * 根据用户名查询用户个数
     * li ning
     * @param userName
     * @return
     */
    public int getUserCountByUserName(String userName);

    /**
     * 批量修改用户的部门(一个部门)
     * li ning
     * @param userIds
     * @param departmentId
     */
    public void changeDepartment(List<String> userIds,String departmentId);

    /**
     * 修改用户的部门（一个部门）
     * li ning
     * @param userId
     * @param departmentId
     */
    public void changeDepartment(String userId,String departmentId);

    /**
     * 获取某个用户的部门列表
     * li ning
     * @param userId
     * @return
     */
    public List<Department> getUserDepartments(String userId);

    /**
     * 为一个用户设置多个部门（原有部门会被清掉）
     * li ning
     * @param userId
     * @param departmentIds
     */
    public void changeDepartment(String userId,List<String> departmentIds);

    /**
     * 为一个用户增加部门（原有的部门保留）
     * li ning
     * @param userId
     * @param departmentId
     */
    public void addDepartment(String userId,String departmentId);

    /**
     * 批量修改用户的角色
     * li ning
     * @param userId
     * @param roleIds
     */
    public void changeRole(String userId,List<String> roleIds);

    /**
     * 设置用户IP访问限制
     * cql
     * @param userId
     * @param Ips
     */
    public void changeIpRange(String userId, List<String> Ips);

    /**
     * 修改用户信息（用户基本信息，部门（一个），角色（多个）,用户访问IP（多个））
     * li ning
     * @param user
     * @param departmentIds
     * @param roleIds
     * @param Ips for example:192.168.1.1-192.168.1.100
     */
    public void updateUser(User user,List<String> departmentIds,List<String> roleIds,List<String> Ips);

    /**
     * 新建用户信息（用户基本信息，部门（一个），角色（多个）,用户访问IP（多个））
     * li ning
     * @param user
     * @param departmentIds
     * @param roleIds
     * @param Ips
     */
    public void insertUser(User user,List<String> departmentIds,List<String> roleIds,List<String> Ips);

    public List<User> getUserListByIds(Set ids);

    /**
     * 根据用户姓名返回用户查询分页结果
     * @param userName 用户姓名
     * @param pageCfg 分页设置
     * @return 查询的分页结果
     */
    public Page<User> findUserPageByUserName(String userName, Pageable pageCfg);

    /**
     * 根据用户所在部门返回用户查询分页结果
     * @param departmentName 用户所在部门
     * @param pageCfg 分页设置
     * @return 查询的分页结果
     */
    public Page<User> findUserPageByDepartment(String departmentName, Pageable pageCfg);

    /**
     * 根据用户角色返回用户查询的分页结果
     * @param roleName 角色名称
     * @param pageCfg 分页设置
     * @return 查询的分页结果
     */
    public Page<User> findUserPageByRole(String roleName, Pageable pageCfg);

    /**
     * 根据部门名称查询部门
     * li ning
     * @param departmentName
     * @return 结果为带层级关系的List
     */
    public List<Department> searchDepartmentByName(String departmentName);

}
