/*
 * Project:  hydroplat-parent
 * Module:   hydroplat-common
 * File:     AuthService.java
 * Modifier: yangxin
 * Modified: 2014-06-12 21:38
 *
 * Copyright (c) 2014 Mapjs All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent
 * or the registration of a utility model, design or code.
 */

package cn.gtmap.egovplat.service;

import cn.gtmap.egovplat.model.auth.Privilege;
import cn.gtmap.egovplat.security.Operation;
import cn.gtmap.egovplat.model.auth.PrivilegeKey;

import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * 权限管理服务.
 * <p/>
 *
 * @author <a href="mailto:yangxin@gtmap.cn">yangxin</a>
 * @version V1.0, 14-6-12
 */
public interface AuthService {

    /**
     * 判断指定用户是否对资源拥有指定操作的权限
     *
     * @param userId    用户id
     * @param nodeId    资源节点id
     * @param operation 操作
     * @return 是否有权限
     */
    boolean isPermitted(String userId, String nodeId, Operation operation);

    /**
     * 判断指定用户是否对资源拥有多个指定操作的权限
     *
     * @param userId     用户id
     * @param nodeId     资源节点id
     * @param operations 操作列表
     * @return 是否有权限
     */
    boolean isPermitted(String userId, String nodeId, Collection<Operation> operations);

    /**
     * 判断指定用户是否对资拥有任意一个指定操作的权限
     *
     * @param userId     用户id
     * @param nodeId     资源节点id
     * @param operations 操作列表
     * @return 是否有权限
     */
    boolean isAnyPermitted(String userId, String nodeId, Collection<Operation> operations);

    /**
     * 获取指定用户对多个资源拥有指定操作的权限的map
     *
     * @param userId    用户id
     * @param nodeIds   资源节点id列表
     * @param operation 操作
     * @return map, key为资源id, value为是否有权限
     */
    Map<String, Boolean> isPermitted(String userId, Collection<String> nodeIds, Operation operation);

    /**
     * 获取指定用户对多个资源拥有多个指定操作的权限map
     *
     * @param userId     用户id
     * @param nodeIds    资源节点id列表
     * @param operations 操作列表
     * @return map, key为资源id, value为是否有权限
     */
    Map<String, Boolean> isPermitted(String userId, Collection<String> nodeIds, Collection<Operation> operations);

    /**
     * 获取一个授权项
     *
     * @param roleId 角色id
     * @param nodeId 资源节点id
     * @return 授权项
     */
    Privilege getPrivilege(String roleId, String nodeId);

    /**
     * 获取一个角色的所有授权项
     *
     * @param roleId   角色id
     * @param nodeType 节点类型,null为所有
     * @return 授权项列表
     */
    List<Privilege> getRolePrivileges(String roleId, Integer nodeType);

    /**
     * 获取一个资源节点的所有授权项
     *
     * @param nodeId 资源节点id
     * @return 授权项列表
     */
    List<Privilege> getNodePrivileges(String nodeId);

    /**
     * 授权
     *
     * @param privileges 授权项列表
     */
    void grant(Collection<Privilege> privileges);

    /**
     * 取消权限
     *
     * @param privileges 授权项列表
     */
    void revoke(Collection<PrivilegeKey> privileges);

    /**
     * 清除指定角色所有授权项
     *
     * @param roleId 角色id
     */
    void cleanRolePrivileges(String roleId);

    /**
     * 清除指定资源节点所有授权项
     *
     * @param nodeId 资源节点id
     */
    void cleanNodePrivileges(String nodeId);
}
