/*
 * Project:  hydroplat-parent
 * Module:   hydroplat-common
 * File:     EntityController.java
 * Modifier: yangxin
 * Modified: 2014-07-01 22:39
 *
 * Copyright (c) 2014 Mapjs All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent
 * or the registration of a utility model, design or code.
 */

package cn.gtmap.egovplat.core.web;

import cn.gtmap.egovplat.core.data.P;
import cn.gtmap.egovplat.core.data.Page;
import cn.gtmap.egovplat.core.data.Pageable;
import cn.gtmap.egovplat.core.entity.IdGetter;
import cn.gtmap.egovplat.core.util.ExUtils;
import cn.gtmap.egovplat.core.util.ClassUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import java.io.Serializable;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:yangxin@gtmap.cn">yangxin</a>
 * @version V1.0, 14-7-1
 */
public abstract class EntityController<E, ID extends Serializable> extends BaseController {
    private final Class<E> entityClass;

    @Autowired(required = false)
    private IdGetter<E, ID> idGetter;

    public Class<E> getEntityClass() {
        return entityClass;
    }

    public EntityController() {
        entityClass = ClassUtils.getGenericParameter0(getClass());
        if (!AopUtils.isAopProxy(this) && entityClass == null) {
            throw new IllegalArgumentException("Entity class not found");
        }
    }

    @ModelAttribute(Vars.MODEL)
    public E getEntity(@RequestParam(value = "id", required = false) ID id, Model model) {
        model.addAttribute(Vars.CONTROLLER, this);
        if (id == null || "".equals(id)) {
            return newEntity();
        } else {
            return loadEntity(id);
        }
    }

    @RequestMapping("list")
    public String list(Model model, HttpServletRequest request, Pageable pageable) {
        model.addAttribute("page", listEntities(request, pageable));
        return getView("list");
    }

    @RequestMapping("ajax-list")
    @ResponseBody
    public Page<E> ajaxList(HttpServletRequest request, Pageable pageable) {
        return listEntities(request, pageable);
    }

    @RequestMapping(value = "edit", method = RequestMethod.GET)
    public String edit(@ModelAttribute(Vars.MODEL) E entity, Model model, HttpServletRequest request) {
        prepareEdit(entity, model, request);
        return getView("list");
    }

    @RequestMapping(value = "edit", method = RequestMethod.POST)
    public String save(@ModelAttribute(Vars.MODEL) E entity, HttpServletRequest request) {
        saveEntity(entity, request);
        return "redirect:" + getView("edit") + "?id=" + getEntityId(entity);
    }

    @RequestMapping(value = "ajax-edit", method = RequestMethod.POST)
    @ResponseBody
    public E AjaxSave(@ModelAttribute(Vars.MODEL) E entity, HttpServletRequest request) {
        saveEntity(entity, request);
        return entity;
    }

    @RequestMapping(value = "remove")
    public String remove(@RequestParam(value = "id") ID[] ids, Pageable pageable) {
        removeEntities(Arrays.asList(ids));
        String view = "redirect:" + getView("list");
        if (pageable != P.DEFAULT) {
            view = view + "?size=" + pageable.getSize() + "&index=" + pageable.getIndex();
        }
        return view;
    }

    @RequestMapping(value = "ajax-remove")
    @ResponseBody
    public Map ajaxRemove(@RequestParam(value = "id") ID[] ids) {
        removeEntities(Arrays.asList(ids));
        return success();
    }

    protected String getView(String action) {
        return "/" + StringUtils.uncapitalize(entityClass.getSimpleName()) + "/" + action;
    }

    protected E newEntity() {
        try {
            return entityClass.newInstance();
        } catch (Exception e) {
            throw ExUtils.wrap(e);
        }
    }

    protected void prepareEdit(E entity, Model model, HttpServletRequest request) {
    }

    protected ID getEntityId(E entity) {
        if (idGetter != null) {
            return idGetter.getId(entity);
        }
        throw new UnsupportedOperationException("getEntityId");
    }

    protected abstract Page<E> listEntities(HttpServletRequest request, Pageable pageable);

    protected abstract E loadEntity(ID id);

    protected abstract void saveEntity(E entity, HttpServletRequest request);

    protected abstract void removeEntities(List<ID> ids);
}
