/*
 * Project:  hydroplat-parent
 * Module:   hydroplat-common
 * File:     SpringWroFilter.java
 * Modifier: yangxin
 * Modified: 2014-06-30 09:46
 *
 * Copyright (c) 2014 Mapjs All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent
 * or the registration of a utility model, design or code.
 */

package cn.gtmap.egovplat.core.support.wro;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;
import ro.isdc.wro.WroRuntimeException;
import ro.isdc.wro.config.Context;
import ro.isdc.wro.config.factory.PropertyWroConfigurationFactory;
import ro.isdc.wro.config.jmx.WroConfiguration;
import ro.isdc.wro.http.WroFilter;
import ro.isdc.wro.http.support.ServletContextAttributeHelper;
import ro.isdc.wro.manager.factory.DefaultWroManagerFactory;
import ro.isdc.wro.manager.factory.WroManagerFactory;
import ro.isdc.wro.model.resource.locator.support.DispatcherStreamLocator;

import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:yangxin@gtmap.cn">yangxin</a>
 * @version V1.0, 14-6-30
 */
public class SpringWroFilter implements Filter {
    private static final Logger LOG = LoggerFactory.getLogger(SpringWroFilter.class);
    private String wroContextPath;
    private FilterConfig filterConfig;
    private WroConfiguration wroConfiguration;
    private WroManagerFactory wroManagerFactory;

    @Override
    public void init(FilterConfig config) throws ServletException {
        filterConfig = config;
        String targetBeanName = config.getInitParameter("targetBeanName");
        if (targetBeanName == null) {
            targetBeanName = "wroConfigurationFactory";
        }
        wroContextPath = config.getInitParameter("wroContextPath");
        if (wroContextPath == null) {
            wroContextPath = "/static/wro";
        }
        wroContextPath = config.getServletContext().getContextPath() + wroContextPath;
        WebApplicationContext ctx = null;
        try {
            ctx = WebApplicationContextUtils.getWebApplicationContext(config.getServletContext());
        } catch (Throwable e) {
            return;
        }
        PropertyWroConfigurationFactory factory = ctx.getBean(targetBeanName, PropertyWroConfigurationFactory.class);
        if (factory == null) {
            throw new WroRuntimeException("Could not locate: " + PropertyWroConfigurationFactory.class.getName() + " instance in applicationContext with bean name: " + targetBeanName);
        }
        wroConfiguration = factory.create();
        wroManagerFactory = DefaultWroManagerFactory.create(factory);
        new ServletContextAttributeHelper(config.getServletContext()).setManagerFactory(wroManagerFactory);
    }

    @Override
    public void doFilter(ServletRequest req, ServletResponse res, FilterChain chain) throws IOException, ServletException {
        if (wroConfiguration == null) {
            chain.doFilter(req, res);
            return;
        }
        HttpServletRequest request = (HttpServletRequest) req;
        HttpServletResponse response = (HttpServletResponse) res;
        try {
            Context.set(Context.webContext(request, response, filterConfig), wroConfiguration);
            String requestURI = request.getRequestURI();
            if (requestURI.startsWith(wroContextPath) &&
                    request.getAttribute(WroFilter.ATTRIBUTE_PASSED_THROUGH_FILTER) == null &&
                    request.getAttribute(DispatcherStreamLocator.ATTRIBUTE_INCLUDED_BY_DISPATCHER) == null) {
                LOG.trace("processing wro request: {}", request.getRequestURI());
                try {
                    request.setAttribute(WroFilter.ATTRIBUTE_PASSED_THROUGH_FILTER, Boolean.TRUE);
                    wroManagerFactory.create().process();
                } catch (final Exception e) {
                    LOG.debug("Wro process error occured", e);
                    chain.doFilter(request, response);
                }
            } else {
                chain.doFilter(request, response);
            }
        } finally {
            Context.unset();
        }
    }

    @Override
    public void destroy() {
        if (wroManagerFactory != null) {
            wroManagerFactory.destroy();
        }
        if (wroConfiguration != null) {
            wroConfiguration.destroy();
        }
        Context.destroy();
    }
}
