/*
 * Project:  hydroplat-parent
 * Module:   hydroplat-common
 * File:     SessionFactoryBean.java
 * Modifier: yangxin
 * Modified: 2014-06-11 10:38
 *
 * Copyright (c) 2014 Mapjs All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent
 * or the registration of a utility model, design or code.
 */

package cn.gtmap.egovplat.core.support.hibernate;

import com.google.common.collect.Sets;
import org.hibernate.SessionFactory;
import org.hibernate.cfg.AvailableSettings;
import org.hibernate.cfg.Environment;
import org.hibernate.engine.spi.SessionFactoryImplementor;
import org.hibernate.event.service.spi.EventListenerGroup;
import org.hibernate.event.service.spi.EventListenerRegistry;
import org.hibernate.event.spi.EventType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.datasource.init.DatabasePopulator;
import org.springframework.jdbc.datasource.init.DatabasePopulatorUtils;
import org.springframework.orm.hibernate4.LocalSessionFactoryBean;
import org.springframework.orm.hibernate4.LocalSessionFactoryBuilder;

import javax.sql.DataSource;
import java.util.Collection;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:yangxin@gtmap.cn">yangxin</a>
 * @version V1.0, 13-7-5
 */
public class SessionFactoryBean extends LocalSessionFactoryBean {
    private static final Logger LOG = LoggerFactory.getLogger(SessionFactoryBean.class);
    private static final Set<String> AUTOS = Sets.newHashSet("update", "create", "create-drop");
    private DatabasePopulator databasePopulator;
    private Map<String, Object> eventListeners;
    private boolean cleaneOneToManyCache = true;

    public void setDatabasePopulator(DatabasePopulator databasePopulator) {
        this.databasePopulator = databasePopulator;
    }

    public void setEventListeners(Map<String, Object> eventListeners) {
        this.eventListeners = eventListeners;
    }

    public void setCleaneOneToManyCache(boolean cleaneOneToManyCache) {
        this.cleaneOneToManyCache = cleaneOneToManyCache;
    }

    @Override
    protected SessionFactory buildSessionFactory(LocalSessionFactoryBuilder sfb) {
        SessionFactory sessionFactory = super.buildSessionFactory(sfb);
        populateDatabase();
        registerEventListeners(sessionFactory);
        return sessionFactory;
    }

    @SuppressWarnings("unchecked")
    private void registerEventListeners(SessionFactory sessionFactory) {
        if (this.eventListeners != null) {
            EventListenerRegistry registry = ((SessionFactoryImplementor) sessionFactory).getServiceRegistry().getService(EventListenerRegistry.class);
            for (Map.Entry<String, Object> entry : this.eventListeners.entrySet()) {
                Object listener = entry.getValue();
                try {
                    EventType eventType = EventType.resolveEventTypeByName(entry.getKey());
                    EventListenerGroup group = registry.getEventListenerGroup(eventType);
                    if (listener instanceof Collection) {
                        for (Object listener1 : (Collection) listener) {
                            group.appendListener(listener1);
                        }
                    } else {
                        group.appendListener(listener);
                    }
                } catch (Exception e) {
                    LOG.error("Unable to register EventListener type:" + entry.getKey(), e);
                }
            }
        }
        if (cleaneOneToManyCache) {
            OneToManyCacheCleaner cleaner = new OneToManyCacheCleaner();
            cleaner.setSessionFactory(sessionFactory);
            cleaner.afterPropertiesSet();
        }
    }

    private void populateDatabase() {
        Properties properties = getConfiguration().getProperties();
        String hbm2ddlAuto = properties.getProperty(AvailableSettings.HBM2DDL_AUTO);
        if (databasePopulator != null && hbm2ddlAuto != null && AUTOS.contains(hbm2ddlAuto)) {
            DatabasePopulatorUtils.execute(this.databasePopulator, (DataSource) properties.get(Environment.DATASOURCE));
        }
    }
}
