/*
 * Project:  hydroplat-parent
 * Module:   hydroplat-common
 * File:     HibernateRepo.java
 * Modifier: yangxin
 * Modified: 2014-06-11 14:40
 *
 * Copyright (c) 2014 Mapjs All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent
 * or the registration of a utility model, design or code.
 */

package cn.gtmap.egovplat.core.support.hibernate;

import cn.gtmap.egovplat.core.data.*;
import cn.gtmap.egovplat.core.data.dsl.DSL;
import cn.gtmap.egovplat.core.ex.EntityNotFoundException;
import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import cn.gtmap.egovplat.core.data.dsl.Builder;
import cn.gtmap.egovplat.core.data.dsl.QueryParam;
import cn.gtmap.egovplat.core.entity.Repo;
import cn.gtmap.egovplat.core.ex.EntityException;
import cn.gtmap.egovplat.core.support.jpa.JPAHelper;
import cn.gtmap.egovplat.core.util.ArrayUtils;
import cn.gtmap.egovplat.core.util.ClassUtils;
import com.mysema.query.dml.DeleteClause;
import com.mysema.query.dml.UpdateClause;
import com.mysema.query.jpa.hibernate.HibernateDeleteClause;
import com.mysema.query.jpa.hibernate.HibernateQuery;
import com.mysema.query.jpa.hibernate.HibernateUpdateClause;
import com.mysema.query.types.EntityPath;
import com.mysema.query.types.Expression;
import com.mysema.query.types.OrderSpecifier;
import com.mysema.query.types.Predicate;
import com.mysema.query.types.path.PathBuilder;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.*;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.engine.spi.SessionImplementor;
import org.hibernate.metadata.ClassMetadata;
import org.hibernate.persister.entity.AbstractEntityPersister;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.querydsl.SimpleEntityPathResolver;
import org.springframework.util.Assert;

import javax.persistence.EntityManagerFactory;
import java.io.Serializable;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oznyang@163.com">oznyang</a>
 * @version V1.0, 13-11-7
 */
@SuppressWarnings("unchecked")
public class HibernateRepo<E, ID extends Serializable> implements Repo<E, ID> {
    private EntityManagerFactory entityManagerFactory;
    private SessionFactory sessionFactory;

    private final Class<E> entityClass;
    private ClassMetadata classMetadata;

    private EntityPath<E> path;
    private PathBuilder<E> pathBuilder;

    public HibernateRepo(Class<E> entityClass) {
        this.entityClass = entityClass;
    }

    public HibernateRepo() {
        entityClass = ClassUtils.getGenericParameter0(getClass());
        if (!AopUtils.isAopProxy(this) && entityClass == null) {
            throw new IllegalArgumentException("Entity class not found");
        }
    }

    @Autowired(required = false)
    public void setSessionFactory(SessionFactory sessionFactory) {
        this.sessionFactory = sessionFactory;
    }

    @Autowired(required = false)
    public void setEntityManagerFactory(EntityManagerFactory entityManagerFactory) {
        this.entityManagerFactory = entityManagerFactory;
    }

    protected SessionFactory getSessionFactory() {
        return entityManagerFactory != null ? JPAHelper.getSessionFactory(entityManagerFactory) : sessionFactory;
    }

    protected Session getSession() {
        return entityManagerFactory != null ? JPAHelper.getSession(entityManagerFactory) : sessionFactory.getCurrentSession();
    }

    @Override
    public Class<E> getEntityClass() {
        return entityClass;
    }

    public E getRaw(ID id) {
        return (E) getSession().get(entityClass, id);
    }

    @Override
    public E get(ID id) {
        return prepare(getRaw(id));
    }

    @Override
    public E load(ID id) throws EntityNotFoundException {
        return assertNotNull(get(id), id);
    }

    @Override
    public boolean exists(ID id) {
        return get(id) != null;
    }

    @Override
    public List<E> list(Iterable<ID> ids) {
        if (ids == null || !ids.iterator().hasNext()) {
            return Collections.emptyList();
        }
        return list(criteria(Restrictions.in(getClassMetadata().getIdentifierPropertyName(), (Collection) ids)));
    }

    @Override
    public Map<ID, E> mget(Iterable<ID> ids) {
        Map<ID, E> map = Maps.newLinkedHashMap();
        for (ID id : ids) {
            E entity = get(id);
            if (entity != null) {
                map.put(id, entity);
            }
        }
        return map;
    }

    @Override
    public E getByNaturalId(String fieldName, Object fieldValue) {
        return getByNaturalId(Collections.singletonMap(fieldName, fieldValue));
    }

    @Override
    public E getByNaturalId(Map<String, Object> naturalIds) {
        NaturalIdLoadAccess nia = getSession().byNaturalId(entityClass);
        for (Map.Entry<String, Object> entry : naturalIds.entrySet()) {
            nia.using(entry.getKey(), entry.getValue());
        }
        return prepare((E) nia.load());
    }

    @Override
    public E loadByNaturalId(String fieldName, Object fieldValue) throws EntityNotFoundException {
        return loadByNaturalId(Collections.singletonMap(fieldName, fieldValue));
    }

    @Override
    public E loadByNaturalId(Map<String, Object> naturalIds) throws EntityNotFoundException {
        return assertNotNull(getByNaturalId(naturalIds), naturalIds);
    }

    @Override
    public boolean exists(Map<String, Object> naturalIds) {
        return getByNaturalId(naturalIds) != null;
    }

    @Override
    public Criteria criteria(Criterion... criterions) {
        return criteria(ArrayUtils.asList(criterions));
    }

    @Override
    public Criteria criteria(Collection<Criterion> criterions) {
        Criteria criteria = getSession().createCriteria(entityClass);
        if (criterions != null) {
            for (Criterion c : criterions) {
                if (c != null) {
                    criteria.add(c);
                }
            }
        }
        return criteria;
    }

    @Override
    public E get(Criteria criteria) {
        return prepare((E) criteria.uniqueResult());
    }

    @Override
    public E load(Criteria criteria) throws EntityNotFoundException {
        return assertNotNull(get(criteria), criteria);
    }

    @Override
    public boolean exists(Criteria criteria) {
        return get(criteria) != null;
    }

    @Override
    public long count(Criteria criteria) {
        return Hibernates.getLong(criteria.setFirstResult(0).setProjection(Projections.rowCount()).uniqueResult());
    }

    @Override
    public List<E> list(Criteria criteria) {
        return prepare(criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY).list());
    }

    @Override
    public Page<E> find(Criteria criteria, Pageable request) {
        if (request == null) {
            request = P.first();
        }
        int size = request.getSize();
        if (size == Pageable.SIZE_ALL) {
            appendOrder(criteria, request);
            return new PageImpl<E>(list(criteria));
        } else if (size == Pageable.SIZE_NONE) {
            return new PageImpl<E>(null, count(criteria), request);
        } else if (size < 0) {
            criteria.setFirstResult(request.getOffset()).setMaxResults(-size);
            appendOrder(criteria, request);
            return new PageImpl<E>(list(criteria));
        }
        criteria.setFirstResult(request.getOffset()).setMaxResults(size);
        appendOrder(criteria, request);
        List<E> items = list(criteria);
        return new PageImpl<E>(items, items.size() == size ? count(criteria) : request.getOffset() + items.size(), request);
    }

    private void appendOrder(Criteria criteria, Pageable request) {
        if (request != null) {
            for (Order order : request.getOrders()) {
                if (order != null) {
                    criteria.addOrder(order.isAsc() ? org.hibernate.criterion.Order.asc(order.getField()) : org.hibernate.criterion.Order.desc(order.getField()));
                }
            }
        }
    }

    @Override
    public Query hql(String hql, Object... args) {
        Query query = getSession().createQuery(hql);
        return addParameters(query, args);
    }

    @Override
    public Query hql(String hql, Map<String, Object> args) {
        Query query = getSession().createQuery(hql);
        return addParameters(query, args);
    }

    @Override
    public SQLQuery entitySql(String sql, Object... args) {
        return sql(sql, args).addEntity(entityClass);
    }

    @Override
    public SQLQuery entitySql(String sql, Map<String, Object> args) {
        return sql(sql, args).addEntity(entityClass);
    }

    @Override
    public SQLQuery sql(String sql, Object... args) {
        return addParameters(getSession().createSQLQuery(sql), args);
    }

    @Override
    public SQLQuery sql(String sql, Map<String, Object> args) {
        return addParameters(getSession().createSQLQuery(sql), args);
    }

    @Override
    public E get(Query query) {
        return prepare((E) query.uniqueResult());
    }

    @Override
    public E load(Query query) throws EntityNotFoundException {
        return assertNotNull(get(query), query);
    }

    @Override
    public boolean exists(Query query) {
        return get(query) != null;
    }

    @Override
    public long count(Query query) {
        return Hibernates.getLong(query.list().size());//.uniqueResult());
    }

    @Override
    public List<E> list(Query query) {
        return prepare(query.list());
    }

    @Override
    public <F, T> List<T> list(Query query, Function<F, T> function) {
        List<F> ret = query.list();
        List<T> list = Lists.newArrayListWithCapacity(ret.size());
        for (F f : ret) {
            list.add(function.apply(f));
        }
        return list;
    }

    @Override
    public Page<E> findByHql(String hql, Pageable request, Object... args) {
        Session session = getSession();
        Query query = session.createQuery(hql);
        Query countQuery = session.createQuery(toCountQueryString(hql));
        addParameters(query, args);
        addParameters(countQuery, args);
        return find(query, countQuery, request);
    }

    @Override
    public Page<E> findByHql(String hql, Map<String, Object> args, Pageable request) {
        Session session = getSession();
        Query query = session.createQuery(hql);
        Query countQuery = session.createQuery(toCountQueryString(hql));
        addParameters(query, args);
        addParameters(countQuery, args);
        return find(query, countQuery, request);
    }

    @Override
    public Page<E> findBySql(String sql, Pageable request, Object... args) {
        Session session = getSession();
        Query query = session.createSQLQuery(sql).addEntity(entityClass);
        Query countQuery = session.createSQLQuery(toCountQueryString(sql));
        addParameters(query, args);
        addParameters(countQuery, args);
        return find(query, countQuery, request);
    }

    @Override
    public Page<E> findBySql(String sql, Map<String, Object> args, Pageable request) {
        Session session = getSession();
        Query query = session.createSQLQuery(sql).addEntity(entityClass);
        Query countQuery = session.createSQLQuery(toCountQueryString(sql));
        addParameters(query, args);
        addParameters(countQuery, args);
        return find(query, countQuery, request);
    }

    @Override
    public <F, T> Page<T> findByHql(String hql, Map<String, Object> args, Pageable request, Function<F, T> function) {
        Page page = (Page) findByHql(hql, args, request);
        List list = page.getItems();
        for (int i = 0, len = list.size(); i < len; i++) {
            list.set(i, function.apply((F) list.get(i)));
        }
        return page;
    }

    @Override
    public <F, T> Page<T> findBySql(String sql, Map<String, Object> args, Pageable request, Function<F, T> function) {
        Session session = getSession();
        Query query = session.createSQLQuery(sql);
        Query countQuery = session.createSQLQuery(toCountQueryString(sql));
        addParameters(query, args);
        addParameters(countQuery, args);
        Page page = find(query, countQuery, request);
        List list = page.getItems();
        for (int i = 0, len = list.size(); i < len; i++) {
            list.set(i, function.apply((F) list.get(i)));
        }
        return page;
    }

    private static final Pattern QL_PATTERN = Pattern.compile("select (.+) (from.+?)", Pattern.DOTALL | Pattern.CASE_INSENSITIVE);
    private static final Pattern ORDERBY_PATTERN = Pattern.compile("order by.+", Pattern.DOTALL | Pattern.CASE_INSENSITIVE);

    private static String toCountQueryString(String qs) {
        int firstChar = qs.charAt(0);
        if (firstChar == 'f' || firstChar == 'F') {
            qs = "select count(*) " + qs;
        } else {
            Matcher matcher = QL_PATTERN.matcher(qs);
            Assert.isTrue(matcher.matches());
            String fields = matcher.group(1);
            String from = matcher.group(2);
            qs = (fields.toLowerCase().contains("distinct") ? "select count(" + fields + ") " : "select count(*) ") + from;
        }
        return ORDERBY_PATTERN.matcher(qs).replaceAll("");
    }

    @Override
    public Page<E> find(Query query, Query countQuery, Pageable request) {
        if (request == null) {
            request = P.first();
        }
        int size = request.getSize();
        if (size == Pageable.SIZE_ALL) {
            return new PageImpl<E>(list(query));
        } else if (size == Pageable.SIZE_NONE) {
            return new PageImpl<E>(null, Hibernates.getInt(countQuery.uniqueResult()), request);
        } else if (size < 0) {
            query.setFirstResult(request.getOffset()).setMaxResults(-size);
            return new PageImpl<E>(list(query));
        }
        query.setFirstResult(request.getOffset()).setMaxResults(size);
        List<E> items = list(query);
        return new PageImpl<E>(items, items.size() == size ? Hibernates.getInt(countQuery.uniqueResult()) : request.getOffset() + items.size(), request);
    }

    @Override
    public Builder q() {
        return DSL.q();
    }

    @Override
    public Builder i() {
        return q().insert();
    }

    @Override
    public Builder u() {
        return q().update();
    }

    @Override
    public Builder d() {
        return q().delete();
    }

    @Override
    public E get(Builder dsl) {
        Query query;
        QueryParam param = dsl.from(entityClass).build();
        if (dsl.isSql()) {
            query = sql(param.getQuery(), param.getArgs());
        } else {
            query = hql(param.getQuery(), param.getArgs());
        }
        return get(query);
    }

    @Override
    public E load(Builder dsl) throws EntityNotFoundException {
        Query query;
        QueryParam param = dsl.from(entityClass).build();
        if (dsl.isSql()) {
            query = sql(param.getQuery(), param.getArgs());
        } else {
            query = hql(param.getQuery(), param.getArgs());
        }
        return assertNotNull(get(query), param.getQuery());
    }

    @Override
    public boolean exists(Builder dsl) {
        return get(dsl) != null;
    }

    @Override
    public long count(Builder dsl) {
        Query query;
        QueryParam param = dsl.from(entityClass).build();
        if (dsl.isSql()) {
            query = sql(param.getCountQuery(), param.getArgs());
        } else {
            query = hql(param.getCountQuery(), param.getArgs());
        }
        return count(query);
    }

    @Override
    public List<E> list(Builder dsl) {
        Query query;
        QueryParam param = dsl.from(entityClass).build();
        if (dsl.isSql()) {
            query = sql(param.getQuery(), param.getArgs());
        } else {
            query = hql(param.getQuery(), param.getArgs());
        }
        return list(query);
    }

    @Override
    public Page<E> find(Builder dsl) {
        Query query;
        Query countQuery;
        QueryParam param = dsl.from(entityClass).build();
        if (dsl.isSql()) {
            query = entitySql(param.getQuery(), param.getArgs());
            countQuery = sql(param.getCountQuery(), param.getArgs());
        } else {
            query = hql(param.getQuery(), param.getArgs());
            countQuery = hql(param.getCountQuery(), param.getArgs());
        }
        return find(query, countQuery, P.offset(param.getOffset(), param.getSize()));
    }

    @Override
    public int execute(Builder dsl) {
        Query query;
        QueryParam param = dsl.from(entityClass).build();
        if (dsl.isSql()) {
            query = sql(param.getQuery(), param.getArgs());
        } else {
            query = hql(param.getQuery(), param.getArgs());
        }
        return query.executeUpdate();
    }

    @Override
    public HibernateQuery dslQuery(Predicate... predicates) {
        HibernateQuery query = new HibernateQuery(getSession()).from(getPath());
        return ArrayUtils.isEmpty(predicates) ? query : query.where(predicates);
    }

    @Override
    public HibernateQuery dslQuery(Collection<Predicate> predicates) {
        return predicates == null ? dslQuery() : dslQuery(predicates.toArray(new Predicate[predicates.size()]));
    }

    @Override
    public UpdateClause<? extends UpdateClause> dslUpdate(Predicate... predicates) {
        HibernateUpdateClause updateClause = new HibernateUpdateClause(getSession(), getPath());
        updateClause.where(predicates);
        return updateClause;
    }

    @Override
    public UpdateClause<? extends UpdateClause> dslUpdate(List<Predicate> predicates) {
        return dslUpdate(predicates.toArray(new Predicate[predicates.size()]));
    }

    @Override
    public DeleteClause<? extends DeleteClause> dslDelete(Predicate... predicates) {
        HibernateDeleteClause deleteClause = new HibernateDeleteClause(getSession(), getPath());
        deleteClause.where(predicates);
        return deleteClause;
    }

    @Override
    public DeleteClause<? extends DeleteClause> dslDelete(Collection<Predicate> predicates) {
        return dslDelete(predicates.toArray(new Predicate[predicates.size()]));
    }

    @Override
    public EntityPath<E> getPath() {
        if (path == null) {
            path = SimpleEntityPathResolver.INSTANCE.createPath(entityClass);
        }
        return path;
    }

    @Override
    public E get(Predicate... predicates) {
        return prepare(dslQuery(predicates).uniqueResult(getPath()));
    }

    @Override
    public E load(Predicate... predicates) throws EntityNotFoundException {
        return assertNotNull(get(predicates), "");
    }

    @Override
    public boolean exists(Predicate... predicates) {
        return get(predicates) != null;
    }

    @Override
    public long count(Predicate... predicates) {
        return dslQuery(predicates).count();
    }

    @Override
    public List<E> list(OrderSpecifier... orders) {
        return list(dslQuery().orderBy(orders));
    }

    @Override
    public List<E> list(Predicate predicate, OrderSpecifier... orders) {
        return list(dslQuery(predicate).orderBy(orders));
    }

    @Override
    public List<E> list(Collection<Predicate> predicates, OrderSpecifier... orders) {
        return list(dslQuery(predicates).orderBy(orders));
    }

    @Override
    public List<E> list(HibernateQuery query) {
        return prepare(query.list(getPath()));
    }

    @Override
    public Page<E> find(Predicate predicate, Pageable request) {
        return find(dslQuery(predicate), request);
    }

    @Override
    public Page<E> find(Collection<Predicate> predicates, Pageable request) {
        return find(dslQuery(predicates), request);
    }

    @Override
    public Page<E> find(Pageable request, Predicate... predicates) {
        return find(ArrayUtils.asList(predicates), request);
    }

    @Override
    public Page<E> find(HibernateQuery query, Pageable request) {
        if (request == null) {
            request = P.first();
        }
        int size = request.getSize();
        if (size == Pageable.SIZE_ALL) {
            for (Order order : request.getOrders()) {
                query.orderBy(toDslOrder(order));
            }
            return new PageImpl<E>(list(query));
        } else if (size == Pageable.SIZE_NONE) {
            return new PageImpl<E>(null, query.count(), request);
        } else if (size < 0) {
            query.offset(request.getOffset()).limit(-size);
            return new PageImpl<E>(list(query));
        }
        query.offset(request.getOffset()).limit(size);
        for (Order order : request.getOrders()) {
            query.orderBy(toDslOrder(order));
        }
        List<E> items = list(query);
        return new PageImpl<E>(items, items.size() == size ? Hibernates.getInt(query.count()) : request.getOffset() + items.size(), request);
    }

    @Override
    public long count() {
        return count(criteria());
    }

    @Override
    public List<E> list() {
        return list(criteria());
    }

    @Override
    public Page<E> find(Pageable request) {
        return find(criteria(), request);
    }

    @Override
    public int execute(Query query) {
        return query.executeUpdate();
    }

    @Override
    public <S extends E> S save(S entity) throws EntityException {
        beforeSave(entity);
        try {
            getSession().persist(entity);
        } catch (Exception e) {
            throw new EntityException(entityClass, e);
        }
        afterSave(entity);
        return prepare(entity);
    }

    @Override
    public <S extends E> List<S> save(Iterable<S> entities) throws EntityException {
        List<S> result = Lists.newArrayList();
        for (S entity : entities) {
            result.add(save(entity));
        }
        return result;
    }

    @Override
    public <S extends E> S merge(S entity) throws EntityException {
        beforeSave(entity);
        try {
            entity = (S) getSession().merge(entity);
        } catch (Exception e) {
            throw new EntityException(entityClass, e);
        }
        afterSave(entity);
        return prepare(entity);
    }

    @Override
    public <S extends E> S saveAndFlush(S entity) throws EntityException {
        S result = save(entity);
        getSession().flush();
        return result;
    }

    @Override
    public void delete(E entity) {
        getSession().delete(beforeDelete(entity));
        afterDelete(entity);
    }

    @Override
    public void delete(Iterable<? extends E> entities) {
        for (E entity : entities) {
            delete(entity);
        }
    }

    @Override
    public void deleteById(ID id) throws EntityNotFoundException {
        delete(load(id));
    }

    @Override
    public void deleteByIds(Iterable<ID> ids) {
        for (ID id : ids) {
            E entity = get(id);
            if (entity != null) {
                delete(entity);
            }
        }
    }

    @Override
    public void deleteByIds(ID... ids) {
        deleteByIds(ArrayUtils.asList(ids));
    }

    @Override
    public void deleteAll() {
        execute(hql("delete from " + getEntityName()));
    }

    @Override
    public boolean accept(Class<?> clazz) {
        return entityClass.isAssignableFrom(clazz);
    }

    public <S extends E> S prepare(S entity) {
        return entity;
    }

    public List<E> prepare(List<E> entities) {
        for (int i = 0, len = entities.size(); i < len; i++) {
            E be = entities.get(i);
            E ae = prepare(be);
            if (ae != be) {
                entities.set(i, ae);
            }
        }
        return entities;
    }

    public E beforeSave(E entity) {
        Assert.notNull(entity, "Entity is required");
        return entity;
    }

    public E afterSave(E entity) {
        return entity;
    }

    public E beforeDelete(E entity) {
        Assert.notNull(entity, "Entity is required");
        return entity;
    }

    public E afterDelete(E entity) {
        return entity;
    }

    protected final ClassMetadata getClassMetadata() {
        if (classMetadata == null) {
            classMetadata = getSessionFactory().getClassMetadata(entityClass);
        }
        return classMetadata;
    }

    protected final ID getId(E entity) {
        return (ID) getClassMetadata().getIdentifier(entity, (SessionImplementor) getSession());
    }

    protected final String getEntityName() {
        return getClassMetadata().getEntityName();
    }

    protected final String getTableName() {
        return ((AbstractEntityPersister) getClassMetadata()).getTableName();
    }

    protected final String getCollectionRole(String collectionName) {
        return getClassMetadata().getEntityName() + "." + collectionName;
    }

    private static <Q extends Query> Q addParameters(Q query, Object... args) {
        if (args != null) {
            for (int i = 0, len = args.length; i < len; i++) {
                query.setParameter(i, args[i]);
            }
        }
        return query;
    }

    private static <Q extends Query> Q addParameters(Q query, Map<String, Object> args) {
        if (args != null) {
            //首先处理Query参数中是否有map中对应的key，只有对应的key才进行赋值操作
            String[] paramters = query.getNamedParameters();
            LinkedHashMap<String,String> paramMap = new LinkedHashMap<String, String>();
            for (int i = 0; i < paramters.length; i++) {
                if (StringUtils.isNoneBlank(paramters[i])){
                    paramMap.put(paramters[i],paramters[i]);
                }
            }
            for (Map.Entry<String, Object> entry : args.entrySet()) {
                Object arg = entry.getValue();
                if (paramMap.containsKey(entry.getKey().toString())){
                    if (arg == null) {
                        query.setParameter(entry.getKey(), null);
                    } else if (arg.getClass().isArray()) {
                        query.setParameterList(entry.getKey(), (Object[]) entry.getValue());
                    } else if (arg instanceof Collection) {
                        query.setParameterList(entry.getKey(), ((Collection) arg));
                    } else {
                        query.setParameter(entry.getKey(), arg);
                    }
                }

            }
        }
        return query;
    }

    protected E assertNotNull(E entity, Object arg) {
        if (entity == null) {
            throw new EntityNotFoundException(entityClass, arg.toString());
        }
        return entity;
    }

    private PathBuilder<E> getBuilder() {
        if (pathBuilder == null) {
            pathBuilder = new PathBuilder<E>(getPath().getType(), getPath().getMetadata());
        }
        return pathBuilder;
    }

    private OrderSpecifier toDslOrder(Order order) {
        Expression<Object> property = getBuilder().get(order.getField());
        return new OrderSpecifier(order.isAsc() ? com.mysema.query.types.Order.ASC : com.mysema.query.types.Order.DESC, property);
    }
}
