/*
 * Project:  hydroplat-parent
 * Module:   hydroplat-common
 * File:     WroDirective.java
 * Modifier: yangxin
 * Modified: 2014-06-30 17:54
 *
 * Copyright (c) 2014 Mapjs All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent
 * or the registration of a utility model, design or code.
 */

package cn.gtmap.egovplat.core.support.freemarker.directive;

import com.google.common.collect.Lists;
import cn.gtmap.egovplat.core.env.Env;
import cn.gtmap.egovplat.core.support.freemarker.FmUtils;
import cn.gtmap.egovplat.core.util.RequestUtils;
import freemarker.core.Environment;
import freemarker.template.TemplateDirectiveBody;
import freemarker.template.TemplateDirectiveModel;
import freemarker.template.TemplateException;
import freemarker.template.TemplateModel;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.context.ServletContextAware;
import ro.isdc.wro.cache.CacheKey;
import ro.isdc.wro.cache.CacheValue;
import ro.isdc.wro.http.support.ServletContextAttributeHelper;
import ro.isdc.wro.manager.WroManager;
import ro.isdc.wro.model.WroModelInspector;
import ro.isdc.wro.model.group.Group;
import ro.isdc.wro.model.resource.Resource;
import ro.isdc.wro.model.resource.ResourceType;

import javax.servlet.ServletContext;
import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:yangxin@gtmap.cn">yangxin</a>
 * @version V1.0, 14-6-30
 */
public class WroDirective implements TemplateDirectiveModel, ServletContextAware {
    private static final Logger LOG = LoggerFactory.getLogger(WroDirective.class);
    public static final String TYPE = "type";
    public static final String EXPLODED = "exploded";
    public static final String MIN = "min";
    public static final String GROUP = "group";
    private ServletContext servletContext;
    private WroManager wroManager;
    private WroModelInspector inspector;

    @Override
    public void setServletContext(ServletContext servletContext) {
        this.servletContext = servletContext;
    }

    public WroManager getWroManager() {
        if (wroManager == null) {
            initWro();
        }
        return wroManager;
    }

    public WroModelInspector getInspector() {
        if (inspector == null) {
            initWro();
        }
        return inspector;
    }

    private void initWro() {
        ServletContextAttributeHelper helper = new ServletContextAttributeHelper(servletContext);
        wroManager = helper.getManagerFactory().create();
        inspector = new WroModelInspector(wroManager.getModelFactory().create());
    }

    @Override
    public void execute(Environment env, Map params, TemplateModel[] loopVars, TemplateDirectiveBody body) throws TemplateException, IOException {
        Object groupObj = FmUtils.getParam(params, GROUP);
        List<String> groups = Lists.newArrayList();
        if (groupObj instanceof Iterable) {
            for (Object obj : (Iterable) groupObj) {
                Collections.addAll(groups, StringUtils.split(obj.toString(), ','));
            }
        } else {
            Collections.addAll(groups, StringUtils.split(groupObj.toString(), ','));
        }
        if (groups.isEmpty()) {
            return;
        }

        String type = FmUtils.getString(params, TYPE);
        boolean minimize = FmUtils.getBoolean(params, MIN, false);
        boolean exploded = FmUtils.getBoolean(params, EXPLODED, true);
        if (Env.isProd()) {
            minimize = true;
            exploded = false;
        }
        StringBuilder jsSb = new StringBuilder(), cssSb = new StringBuilder();
        for (String s : groups) {
            Group group = getInspector().getGroupByName(s.trim());
            if (group != null) {
                if (exploded) {
                    for (Resource resource : group.getResources()) {
                        String url = resource.getUri();
                        include(resource.getType() == ResourceType.CSS ? cssSb : jsSb, resource.getType(), url, null, null);
                    }
                } else {
                    CacheKey key = new CacheKey(group.getName(), ResourceType.JS, minimize);
                    CacheValue cacheValue = getWroManager().getCacheStrategy().get(key);
                    String version = cacheValue != null ? cacheValue.getHash() : null;
                    if ("js".equals(type)) {
                        include(jsSb, ResourceType.JS, "/static/wro/" + group.getName() + ".js", version, minimize);
                    } else if ("css".equals(type)) {
                        include(cssSb, ResourceType.CSS, "/static/wro/" + group.getName() + ".css", version, minimize);
                    } else {
                        include(cssSb, ResourceType.CSS, "/static/wro/" + group.getName() + ".css", version, minimize);
                        include(jsSb, ResourceType.JS, "/static/wro/" + group.getName() + ".js", version, minimize);
                    }
                }
            } else {
                LOG.info("Wro group [{}] not found", s);
            }
        }
        if (cssSb.length() > 0) {
            env.getOut().write(cssSb.toString());
        }
        if (jsSb.length() > 0) {
            env.getOut().write(jsSb.toString());
        }
    }

    private void include(StringBuilder sb, ResourceType type, String url, String version, Boolean minimize) {
        sb.append(type == ResourceType.CSS ? "<link href=\"" : "<script src=\"");
        sb.append(RequestUtils.buildUrl(url, null));
        if (minimize != null) {
            if (minimize) {
                if (version != null) {
                    sb.append("?").append(StringUtils.left(version, 8));
                }
            } else {
                sb.append("?minimize=false");
                if (version != null) {
                    sb.append("&").append(StringUtils.left(version, 8));
                }
            }
        }
        sb.append(type == ResourceType.CSS ? "\" media=\"all\" rel=\"stylesheet\" type=\"text/css\"/>\n" : "\" type=\"text/javascript\"></script>\n");
    }
}
