/*
 * Project:  hydroplat-parent
 * Module:   hydroplat-common
 * File:     DefaultSourceExtractor.java
 * Modifier: yangxin
 * Modified: 2014-07-02 19:19
 *
 * Copyright (c) 2014 Mapjs All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent
 * or the registration of a utility model, design or code.
 */

package cn.gtmap.egovplat.core.ex.impl;

import cn.gtmap.egovplat.core.ex.Source;
import cn.gtmap.egovplat.core.ex.SourceExtractor;
import cn.gtmap.egovplat.core.util.Charsets;
import cn.gtmap.egovplat.core.util.ExUtils;
import cn.gtmap.egovplat.core.util.FileUtils;
import com.google.common.collect.Sets;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.util.ClassUtils;
import org.springframework.util.ResourceUtils;

import java.io.File;
import java.io.IOException;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:yangxin@gtmap.cn">yangxin</a>
 * @version V1.0, 14-7-2
 */
public class DefaultSourceExtractor implements SourceExtractor<Throwable> {
    private String appPackage;
    private Set<String> excludeClasses = Sets.newHashSet();
    private String[] sourcePaths = new String[]{"src/main/java", "../common/src/main/java"};

    public void setAppPackage(String appPackage) {
        this.appPackage = appPackage;
    }

    public void setExcludeClasses(Set<String> excludeClasses) {
        this.excludeClasses.addAll(excludeClasses);
    }

    public void setSourcePaths(String[] sourcePaths) {
        this.sourcePaths = sourcePaths;
    }

    @Override
    public boolean isSupport(Throwable ex) {
        return true;
    }

    @Override
    public Source extract(Throwable ex) {
        Source source = new Source();
        StackTraceElement trace = null;
        for (StackTraceElement ste : ex.getStackTrace()) {
            String name = ste.getClassName();
            if (trace == null) {
                trace = ste;
            }
            if (excludeClasses.contains(name)) {
                continue;
            }
            if (name.startsWith(appPackage)) {
                trace = ste;
                break;
            }
        }
        if (trace != null) {
            source.setFile(trace.getFileName());
            source.setLineNumber(trace.getLineNumber());
            source.setTitle("Java exception");
            readLines(source, trace);
        }
        source.setDescription(ExUtils.buildNestedMessage(ex));
        return source;
    }

    private void readLines(Source source, StackTraceElement trace) {
        String path = StringUtils.replace(trace.getClassName(), ".", "/") + ".java";
        for (String sp : sourcePaths) {
            File file = new File(sp, path);
            if (file.exists()) {
                try {
                    source.setFile(file.getPath());
                    source.setLines(FileUtils.readLines(file, Charsets.UTF8));
                    return;
                } catch (IOException ignored) {
                }
            }
        }
        try {
            Class clazz = ClassUtils.forName(trace.getClassName(), ClassUtils.getDefaultClassLoader());
            String jarPath = clazz.getProtectionDomain().getCodeSource().getLocation().toString();
            String sourcePath = StringUtils.substringBeforeLast(jarPath, ".") + "-sources.jar";
            File sourceFile = ResourceUtils.getFile(sourcePath);
            if (sourceFile.exists()) {
                try {
                    ZipFile zf = new ZipFile(sourceFile);
                    ZipEntry ze = zf.getEntry(path);
                    if (ze != null) {
                        source.setFile(path);
                        source.setLines(IOUtils.readLines(zf.getInputStream(ze), Charsets.UTF8));
                    }
                } catch (IOException ignored) {
                }
            }
        } catch (Exception ignored) {
        }
    }
}
