/*
 * Project:  hydroplat-parent
 * Module:   hydroplat-common
 * File:     Repo.java
 * Modifier: yangxin
 * Modified: 2014-06-11 14:40
 *
 * Copyright (c) 2014 Mapjs All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent
 * or the registration of a utility model, design or code.
 */

package cn.gtmap.egovplat.core.entity;

import cn.gtmap.egovplat.core.data.Page;
import cn.gtmap.egovplat.core.data.Pageable;
import cn.gtmap.egovplat.core.ex.EntityNotFoundException;
import com.google.common.base.Function;
import cn.gtmap.egovplat.core.data.dsl.Builder;
import cn.gtmap.egovplat.core.ex.EntityException;
import com.mysema.query.dml.DeleteClause;
import com.mysema.query.dml.UpdateClause;
import com.mysema.query.jpa.hibernate.HibernateQuery;
import com.mysema.query.types.OrderSpecifier;
import com.mysema.query.types.Path;
import com.mysema.query.types.Predicate;
import org.hibernate.Criteria;
import org.hibernate.Query;
import org.hibernate.SQLQuery;
import org.hibernate.criterion.Criterion;

import java.io.Serializable;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:yangxin@gtmap.cn">yangxin</a>
 * @version V1.0, 13-6-20
 */
public interface Repo<E, ID extends Serializable> extends EntityFilter<E> {

    /**
     * 获取repo关联的实体类
     *
     * @return 实体类
     */
    Class<E> getEntityClass();

    /**
     * 获取原生实体对象
     *
     * @param id 主键
     * @return 实体对象
     */
    E getRaw(ID id);

    /**
     * 根据主键获取对象,如果数据库没有对应记录则返回null
     * 优先使用下面的load方法
     *
     * @param id 主键
     * @return 实体类对象
     */
    E get(ID id);

    /**
     * 根据主键获取对象，如果数据库没有对应记录则抛实体类没找到异常
     *
     * @param id 主键
     * @return 实体类对象
     * @throws cn.gtmap.egovplat.core.ex.EntityNotFoundException 实体对象没有找到
     */
    E load(ID id) throws EntityNotFoundException;

    /**
     * 根据主键判断该对象是否存在
     *
     * @param id 主键
     * @return 实体对象是否存在
     */
    boolean exists(ID id);

    /**
     * 根据主键列表批量获取对象
     *
     * @param ids 主键列表迭代对象
     * @return 实体对象列表
     */
    List<E> list(Iterable<ID> ids);

    /**
     * 根据主键列表批量获取对象
     *
     * @param ids 主键列表迭代对象
     * @return 实体对象map
     */
    Map<ID, E> mget(Iterable<ID> ids);

    /**
     * 根据自然键获取对象
     *
     * @param fieldName  自然键字段名,对应model中用@NaturalId注解的字段
     * @param fieldValue 自然键字段值
     * @return 实体类对象
     */
    E getByNaturalId(String fieldName, Object fieldValue);

    /**
     * 根据复合自然键map获取对象
     *
     * @param naturalIds 自然键map组合 key为自然键字段，value为自然键字段值
     * @return 实体类对象
     */
    E getByNaturalId(Map<String, Object> naturalIds);

    /**
     * 根据自然键获取对象，如果查询为空则抛不存在异常
     *
     * @param fieldName  自然键字段名
     * @param fieldValue 自然键字段值
     * @return 实体类对象
     * @throws EntityNotFoundException 实体对象没有找到
     */
    E loadByNaturalId(String fieldName, Object fieldValue) throws EntityNotFoundException;

    /**
     * 根据复合自然键map获取对象，如果查询为空则抛不存在异常
     *
     * @param naturalIds 自然键map组合 key为自然键字段，value为自然键字段值，需要包含model所有中用@NaturalId注解的字段
     * @return 实体类对象
     * @throws EntityNotFoundException 实体对象没有找到
     */
    E loadByNaturalId(Map<String, Object> naturalIds) throws EntityNotFoundException;

    /**
     * 根据复合自然键map判断该对象是否存在
     *
     * @param naturalIds 自然键map组合 key为自然键字段，value为自然键字段值，需要包含model所有中用@NaturalId注解的字段
     * @return 实体对象是否存在
     */
    boolean exists(Map<String, Object> naturalIds);

    /**
     * 根据criterion条件数组获取Criteria结果
     *
     * @param criterions 查询criteria条件数组
     * @return Criteria 条件对象
     */
    Criteria criteria(Criterion... criterions);

    /**
     * 根据criterion条件集合获取Criteria结果
     *
     * @param criterions 查询criteria条件集合
     * @return Criteria  条件对象
     */
    Criteria criteria(Collection<Criterion> criterions);

    /**
     * 根据criteria获取实体类对象
     *
     * @param criteria criteria
     * @return 实体类对象
     */
    E get(Criteria criteria);

    /**
     * 根据criteria获取实体类对象,如果查询为空则抛不存在异常
     *
     * @param criteria 查询criteria
     * @return 实体类对象
     * @throws EntityNotFoundException
     */
    E load(Criteria criteria) throws EntityNotFoundException;

    /**
     * 根据criteria判断该对象是否存在
     *
     * @param criteria 查询criteria
     * @return 布尔类型
     */
    boolean exists(Criteria criteria);

    /**
     * 根据criteria查询满足条件的记录总数
     *
     * @param criteria 查询criteria
     * @return 记录总数
     */
    long count(Criteria criteria);

    /**
     * 根据criteria查询满足条件的记录集合
     *
     * @param criteria 查询criteria
     * @return 实体类集合
     */
    List<E> list(Criteria criteria);

    /**
     * 根据criteria和分页参数，获取分页对象
     *
     * @param criteria 查询criteria
     * @param request  分页参数
     * @return 分页对象
     */
    Page<E> find(Criteria criteria, Pageable request);

    /**
     * 根据hql语句和对应的参数，获取Query对象
     *
     * @param hql  hql语句,参数使用=?、=?、=?...
     * @param args 查询参数，对应顺序...
     * 如果需要like等操作，则需要在args参数中加“%”处理，而不是在hql语句中
     * @return Query对象
     */
    Query hql(String hql, Object... args);

    /**
     * 根据hql语句和对应的参数，获取Query对象
     *
     * @param hql  hql语句,参数使用=:field1、:field2、:field3...
     * @param args 查询参数，map类型，key对应对应field，value对应field对应的值
     * @return Query对象
     */
    Query hql(String hql, Map<String, Object> args);

    /**
     * 根据sql语句和对应的参数，获取当前实体类的SQLQuery对象
     *
     * @param sql  sql语句,参数使用=?、=?、=?...
     * @param args 查询参数，对应0，1，2...
     * 如果需要like等操作，则需要在args参数中加“%”处理，而不是在hql语句中
     * @return SQLQuery对象
     */
    SQLQuery entitySql(String sql, Object... args);

    /**
     * 根据sql语句和对应的参数，获取实体类的SQLQuery对象
     *
     * @param sql  sql语句,参数使用=:field1、:field2、:field3...
     * @param args 查询参数，map类型，key对应对应field，value对应field对应的值
     * @return SQLQuery对象
     */
    SQLQuery entitySql(String sql, Map<String, Object> args);

    /**
     * 根据sql语句和对应的参数，获取SQLQuery对象
     *
     * @param sql  sql语句,参数使用=?、=?、=?...
     * @param args 查询参数，对应0，1，2...
     * 如果需要like等操作，则需要在args参数中加“%”处理，而不是在hql语句中
     * @return SQLQuery对象
     */
    SQLQuery sql(String sql, Object... args);

    /**
     * 根据sql语句和对应的参数，获取SQLQuery对象
     *
     * @param sql  sql语句,参数使用=:field1、:field2、:field3...
     * @param args 查询参数，map类型，key对应对应field，value对应field对应的值
     * @return SQLQuery对象
     */
    SQLQuery sql(String sql, Map<String, Object> args);

    /**
     * 根据query获取实体类对象
     *
     * @param query 查询query
     * @return 实体类对象
     */
    E get(Query query);

    /**
     * 根据query获取实体类对象,如果查询为空则抛不存在异常
     * 优先使用
     *
     * @param query 查询query
     * @return 实体类对象
     * @throws EntityNotFoundException
     */
    E load(Query query) throws EntityNotFoundException;

    /**
     * 根据query判断该对象是否存在
     *
     * @param query 查询query
     * @return 布尔类型
     */
    boolean exists(Query query);

    /**
     * 根据query查询满足条件的记录总数
     *
     * @param query 查询query
     * @return 记录总数
     */
    long count(Query query);

    /**
     * 根据query查询满足条件的记录集合
     *
     * @param query 查询query
     * @return 实体类集合
     */
    List<E> list(Query query);

    /**
     * 根据query查询,并使用function做转换
     *
     * @param query    查询query
     * @param function 转换函数
     * @param <F>      输入
     * @param <T>      输出
     * @return 转换完的对象集合
     */
    <F, T> List<T> list(Query query, Function<F, T> function);

    /**
     * 根据hql做分页查询
     *
     * @param hql     hql语句
     * @param args    查询参数
     * @param request 分页参数
     * @return 分页对象
     */
    Page<E> findByHql(String hql, Map<String, Object> args, Pageable request);

    /**
     * 根据hql做分页查询
     *
     * @param hql     hql语句
     * @param args    查询参数
     * @param request 分页参数
     * @return 分页对象
     */
    Page<E> findByHql(String hql, Pageable request, Object... args);

    /**
     * 根据sql做分页查询
     *
     * @param sql     sql语句
     * @param args    查询参数
     * @param request 分页参数
     * @return 分页对象
     */
    Page<E> findBySql(String sql, Map<String, Object> args, Pageable request);

    /**
     * 根据hql做分页查询
     *
     * @param sql     sql语句
     * @param request 分页参数
     * @param args    查询参数
     * @return 分页对象
     */
    Page<E> findBySql(String sql, Pageable request, Object... args);

    /**
     * 根据hql做分页查询,并使用function做转换
     *
     * @param hql      hql语句
     * @param args     查询参数
     * @param request  分页参数
     * @param function 转换函数
     * @param <F>      输入
     * @param <T>      输出
     * @return 转换完的分页对象
     */
    <F, T> Page<T> findByHql(String hql, Map<String, Object> args, Pageable request, Function<F, T> function);

    /**
     * 根据sql做分页查询,并使用function做转换
     *
     * @param sql      sql语句
     * @param args     查询参数
     * @param request  分页参数
     * @param function 转换函数
     * @param <F>      输入
     * @param <T>      输出
     * @return 转换完的分页对象
     */
    <F, T> Page<T> findBySql(String sql, Map<String, Object> args, Pageable request, Function<F, T> function);

    /**
     * 根据查询query、总数query和分页参数获取实体类分页对象
     *
     * @param query      查询query
     * @param countQuery 总数query
     * @param request    分页参数
     * @return 分页对象
     */
    Page<E> find(Query query, Query countQuery, Pageable request);

    /**
     * 获取Builder对象
     *
     * @return Builder对象
     */
    Builder q();

    /**
     * 执行Builder的插入操作
     *
     * @return Builder对象
     */
    Builder i();

    /**
     * 执行Builder的插入操作
     *
     * @return Builder对象
     */
    Builder u();

    /**
     * 执行Builder的插入操作
     *
     * @return Builder对象
     */
    Builder d();

    /**
     * 根据builder获取实体类对象
     *
     * @param dsl dsl表达式
     * @return 实体类对象
     */
    E get(Builder dsl);

    /**
     * 根据builder获取实体类对象，如果查询为空则抛不存在异常
     * 优先使用load方法
     *
     * @param dsl dsl表达式
     * @return 实体类对象
     */
    E load(Builder dsl) throws EntityNotFoundException;

    /**
     * 根据builder判断是否存在实体类对象
     *
     * @param dsl dsl表达式
     * @return 布尔类型
     */
    boolean exists(Builder dsl);

    /**
     * 根据builder查询满足条件的记录总数
     *
     * @param dsl dsl表达式
     * @return 记录总数
     */
    long count(Builder dsl);

    /**
     * 根据builder查询满足条件的记录集合
     *
     * @param dsl dsl表达式
     * @return 实体类集合
     */
    List<E> list(Builder dsl);

    /**
     * 根据builder获取实体类分页对象
     *
     * @param dsl dsl表达式
     * @return 分页对象
     */
    Page<E> find(Builder dsl);

    /**
     * 执行builder
     * query.executeUpdate()
     *
     * @param dsl dsl表达式
     * @return 执行状态
     */
    int execute(Builder dsl);

    /**
     * 根据predicates查询组合获取HibernateQuery对象
     *
     * @param predicates 查询组合predicates
     * @return querydsl查询对象
     */
    HibernateQuery dslQuery(Predicate... predicates);

    /**
     * 根据predicates查询集合获取HibernateQuery对象
     *
     * @param predicates 查询集合predicates
     * @return querydsl查询对象
     */
    HibernateQuery dslQuery(Collection<Predicate> predicates);

    /**
     * 根据predicates查询组合返回UpdateClause
     *
     * @param predicates 查询组合predicates
     * @return querydsl更新对象
     */
    UpdateClause<? extends UpdateClause> dslUpdate(Predicate... predicates);

    /**
     * 根据predicates查询集合返回UpdateClause
     *
     * @param predicates 查询集合predicates
     * @return querydsl更新对象
     */
    UpdateClause<? extends UpdateClause> dslUpdate(List<Predicate> predicates);

    /**
     * 根据predicates查询组合返回DeleteClause
     *
     * @param predicates 查询组合predicates
     * @return querydsl删除对象
     */
    DeleteClause<? extends DeleteClause> dslDelete(Predicate... predicates);

    /**
     * 根据predicates查询集合返回DeleteClause
     *
     * @param predicates 查询集合predicates
     * @return querydsl删除对象
     */
    DeleteClause<? extends DeleteClause> dslDelete(Collection<Predicate> predicates);

    Path getPath();

    /**
     * 根据predicates查询组合返回实体类对象
     *
     * @param predicates 查询组合predicates
     * @return 实体类对象
     */
    E get(Predicate... predicates);

    /**
     * 根据predicates查询组合获取实体类对象，如果查询为空则抛不存在异常
     * 优先使用load方法
     *
     * @param predicates 查询组合predicates
     * @return 实体类对象
     * @throws EntityNotFoundException
     */
    E load(Predicate... predicates) throws EntityNotFoundException;

    /**
     * 根据predicates查询组合判断是否存在实体类对象
     *
     * @param predicates 查询组合predicates
     * @return 布尔类型
     */
    boolean exists(Predicate... predicates);

    /**
     * 根据predicates查询组合查询满足条件的记录总数
     *
     * @param predicates 查询组合predicates
     * @return 记录总数
     */
    long count(Predicate... predicates);

    /**
     * 根据OrderSpecifier组合获取实体类对象集合
     *
     * @param orders 排序组合orders
     * @return 实体类对象集合
     */
    List<E> list(OrderSpecifier... orders);

    /**
     * 根据predicates查询组合和OrderSpecifier组合获取实体类对象集合
     *
     * @param predicate 查询组合predicates
     * @param orders    排序组合orders
     * @return 实体类对象集合
     */
    List<E> list(Predicate predicate, OrderSpecifier... orders);

    /**
     * 根据predicates查询集合和OrderSpecifier组合获取实体类对象集合
     *
     * @param predicates 查询集合predicates
     * @param orders     排序组合orders
     * @return 实体类对象集合
     */
    List<E> list(Collection<Predicate> predicates, OrderSpecifier... orders);

    /**
     * 根据HibernateQuery获取实体类集合
     *
     * @param query 查询query
     * @return 实体类对象集合
     */
    List<E> list(HibernateQuery query);

    /**
     * 根据查询predicate和分页参数request获取分页对象
     *
     * @param predicate 查询predicate
     * @param request   分页参数request
     * @return 分页对象
     */
    Page<E> find(Predicate predicate, Pageable request);

    /**
     * 根据查询集合predicate和分页参数request获取分页对象
     *
     * @param predicates 查询集合predicate
     * @param request    分页参数request
     * @return 分页对象
     */
    Page<E> find(Collection<Predicate> predicates, Pageable request);

    /**
     * 根据分页参数request和查询集合predicates获取分页对象
     *
     * @param request    分页参数request
     * @param predicates 查询集合predicates
     * @return 分页对象
     */
    Page<E> find(Pageable request, Predicate... predicates);

    /**
     * 根据HibernateQuery和分页参数request获取分页对象
     *
     * @param query   查询query
     * @param request 分页参数request
     * @return 分页对象
     */
    Page<E> find(HibernateQuery query, Pageable request);

    /**
     * 获取记录总数（无条件）
     *
     * @return 记录总数
     */
    long count();

    /**
     * 获取实体类对象集合（无条件）
     *
     * @return 实体类对象集合
     */
    List<E> list();

    /**
     * 根据分页参数获取分页对象（无条件）
     *
     * @param request 分页参数request
     * @return 分页对象
     */
    Page<E> find(Pageable request);

    /**
     * 执行query查询
     *
     * @param query 查询Query
     * @return 执行状态
     */
    int execute(Query query);

    /**
     * 保存实体类对象
     *
     * @param entity 实体类对象
     * @param <S>    实体类对象类
     * @return 实体类对象
     * @throws EntityException
     */
    <S extends E> S save(S entity) throws EntityException;

    /**
     * 保存实体类对象集合
     *
     * @param entities 实体类对象集合
     * @param <S>      实体类对象类
     * @return 实体类对象集合
     * @throws EntityException
     */
    <S extends E> List<S> save(Iterable<S> entities) throws EntityException;

    /**
     * 保存托管状态的实体类对象
     *
     * @param entity 实体类对象
     * @param <S>    实体类对象类
     * @return 实体类对象
     * @throws EntityException
     */
    <S extends E> S merge(S entity) throws EntityException;

    /**
     * 将实体类操作的session缓存中的数据与数据库同步
     *
     * @param entity 实体类对象
     * @param <S>    实体类对象类
     * @return 实体类对象
     * @throws EntityException
     */
    <S extends E> S saveAndFlush(S entity) throws EntityException;

    /**
     * 删除实体类对象
     *
     * @param entity 实体类对象
     */
    void delete(E entity);

    /**
     * 删除实体类对象集合
     *
     * @param entities 实体类对象集合
     */
    void delete(Iterable<? extends E> entities);

    /**
     * 根据主键删除实体类对象
     *
     * @param id 主键
     * @throws EntityNotFoundException
     */
    void deleteById(ID id) throws EntityNotFoundException;

    /**
     * 根据主键集合删除符合条件的实体类
     *
     * @param ids 主键集合
     */
    void deleteByIds(Iterable<ID> ids);

    /**
     * 根据主键组合删除符合条件的实体类
     *
     * @param ids 主键组合
     */
    void deleteByIds(ID... ids);

    /**
     * 删除所有记录
     */
    void deleteAll();
}
