/*
 * Project:  hydroplat-parent
 * Module:   hydroplat-common
 * File:     CriterionImpl.java
 * Modifier: yangxin
 * Modified: 2014-06-11 10:38
 *
 * Copyright (c) 2014 Mapjs All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent
 * or the registration of a utility model, design or code.
 */

package cn.gtmap.egovplat.core.data.dsl;

import com.google.common.collect.BoundType;
import com.google.common.collect.Range;
import cn.gtmap.egovplat.core.util.ArrayUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.LinkedList;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:yangxin@gtmap.cn">yangxin</a>
 * @version V1.0, 13-7-2
 */
@SuppressWarnings({"rawtypes"})
final class CriterionImpl implements Criterion<Criterion> {

    static enum Type {
        AND, OR, EXPR, NULL, NOT_NULL, EQ, NE, LIKE, GT, LT, GE, LE, BETWEEN, IN, NOT_IN, EXISTS, NOT_EXISTS
    }

    Type type;
    String name;
    Object value;
    LinkedList<Criterion> children;

    CriterionImpl() {
    }

    CriterionImpl(Type type, String name, Object value) {
        this.type = type;
        this.name = name;
        this.value = value;
    }

    @Override
    public Criterion and(Criterion... criterions) {
        return and(ArrayUtils.asList(criterions));
    }

    @Override
    public Criterion and(Collection<Criterion> criterions) {
        return withCriterions(Type.AND, criterions);
    }

    @Override
    public Criterion or(Criterion... criterions) {
        return or(ArrayUtils.asList(criterions));
    }

    @Override
    public Criterion or(Collection<Criterion> criterions) {
        return withCriterions(Type.OR, criterions);
    }

    @Override
    public Criterion expr(String expr) {
        return StringUtils.isEmpty(expr) ? this : with(Type.EXPR, expr.trim(), null);
    }

    @Override
    public Criterion expr(String... exprs) {
        return expr(ArrayUtils.asList(exprs));
    }

    @Override
    public Criterion expr(Collection<String> exprs) {
        if (exprs != null && exprs.size() > 0) {
            for (String expr : exprs) {
                expr(expr);
            }
        }
        return this;
    }

    @Override
    public Criterion isNull(String name) {
        return with(Type.NULL, name, null);
    }

    @Override
    public Criterion notNull(String name) {
        return with(Type.NOT_NULL, name, null);
    }

    @Override
    public Criterion empty(String name) {
        return eq(name, "");
    }

    @Override
    public Criterion notEmpty(String name) {
        return ne(name, "");
    }

    @Override
    public Criterion eq(String name, Object value) {
        return with(Type.EQ, name, value);
    }

    @Override
    public Criterion eqIf(String name, Object value, boolean... conditions) {
        if (!ArrayUtils.isEmpty(conditions)) {
            for (boolean condition : conditions) {
                if (!condition) {
                    return this;
                }
            }
        } else if (value == null) {
            return this;
        }
        return eq(name, value);
    }

    @Override
    public Criterion eqIfHasValue(String name, Object value) {
        if (value == null) {
            return this;
        } else if (value instanceof String) {
            if (StringUtils.isBlank((String) value)) {
                return this;
            }
        } else if (value instanceof Number) {
            if (((Number) value).longValue() == 0) {
                return this;
            }
        } else if (value instanceof Date) {
            if (((Date) value).getTime() < 1) {
                return this;
            }
        }
        return eq(name, value);
    }

    @Override
    public Criterion eqOrNull(String name, Object value) {
        return value == null ? isNull(name) : eq(name, value);
    }

    @Override
    public Criterion ne(String name, Object value) {
        return with(Type.NE, name, value);
    }

    @Override
    public Criterion neIf(String name, Object value) {
        return value == null ? this : ne(name, value);
    }

    @Override
    public Criterion neOrNotNull(String name, Object value) {
        return value == null ? notNull(name) : ne(name, value);
    }

    @Override
    public Criterion like(String name, String value) {
        return like(name, value, MatchMode.ANYWHERE);
    }

    @Override
    public Criterion likeIf(String name, String value) {
        return value == null ? this : like(name, value);
    }

    @Override
    public Criterion like(String name, String value, MatchMode matchMode) {
        return with(Type.LIKE, name, matchMode.toMatchString(value));
    }

    @Override
    public Criterion likeIf(String name, String value, MatchMode matchMode) {
        return value == null ? this : like(name, value, matchMode);
    }

    @Override
    public Criterion gt(String name, Object value) {
        return with(Type.GT, name, value);
    }

    @Override
    public Criterion gtIf(String name, Object value) {
        return value == null ? this : gt(name, value);
    }

    @Override
    public Criterion lt(String name, Object value) {
        return with(Type.LT, name, value);
    }

    @Override
    public Criterion ltIf(String name, Object value) {
        return value == null ? this : lt(name, value);
    }

    @Override
    public Criterion ge(String name, Object value) {
        return with(Type.GE, name, value);
    }

    @Override
    public Criterion geIf(String name, Object value) {
        return value == null ? this : ge(name, value);
    }

    @Override
    public Criterion le(String name, Object value) {
        return with(Type.LE, name, value);
    }

    @Override
    public Criterion leIf(String name, Object value) {
        return value == null ? this : le(name, value);
    }

    @Override
    public Criterion between(String name, Object lo, Object hi) {
        return with(Type.BETWEEN, name, new Object[]{lo, hi});
    }

    @Override
    public Criterion between(String name, Range range) {
        if (range != null) {
            if (range.hasLowerBound()) {
                if (range.lowerBoundType() == BoundType.CLOSED) {
                    if (range.hasUpperBound() && range.upperBoundType() == BoundType.CLOSED) {
                        between(name, range.lowerEndpoint(), range.upperEndpoint());
                        return this;
                    }
                    ge(name, range.lowerEndpoint());
                } else {
                    gt(name, range.lowerEndpoint());
                }
            }

            if (range.hasUpperBound()) {
                if (range.upperBoundType() == BoundType.CLOSED) {
                    le(name, range.upperEndpoint());
                } else {
                    lt(name, range.upperEndpoint());
                }
            }
        }
        return this;
    }

    @Override
    public Criterion in(String name, Object value) {
        return with(Type.IN, name, value);
    }

    @Override
    public Criterion in(String name, Object... values) {
        return in(name, ArrayUtils.asList(values));
    }

    @Override
    public Criterion in(String name, Collection values) {
        return values != null && !values.isEmpty() ? with(Type.IN, name, values) : this;
    }

    @Override
    public Criterion notIn(String name, Object value) {
        return with(Type.NOT_IN, name, value);
    }

    @Override
    public Criterion notIn(String name, Object... values) {
        return notIn(name, ArrayUtils.asList(values));
    }

    @Override
    public Criterion notIn(String name, Collection values) {
        return values != null && !values.isEmpty() ? with(Type.NOT_IN, name, values) : this;
    }

    @Override
    public Criterion exists(Object value) {
        return with(Type.EXISTS, "", value);
    }

    @Override
    public Criterion notExists(Object value) {
        return with(Type.NOT_EXISTS, "", value);
    }

    private boolean islogical() {
        return type == Type.AND || type == Type.OR;
    }

    private Criterion copy() {
        CriterionImpl criterion = new CriterionImpl(type, name, value);
        criterion.children = children;
        return criterion;
    }

    Criterion with(Type otherType, String name, Object value) {
        if (type == null) {
            type = otherType;
            this.name = name;
            this.value = value;
        } else {
            CriterionImpl other = new CriterionImpl(otherType, name, value);
            if (islogical()) {
                children.add(other);
            } else {
                children = newLinkedList(copy(), other);
                type = Type.AND;
            }
        }
        return this;
    }

    Criterion withCriterions(Type otherType, Collection<Criterion> criterions) {
        if (otherType == null) {
            otherType = Type.AND;
        }
        if (type == null) {
            type = Type.AND;
        }
        if (type != otherType && (!islogical() || children != null && children.size() > 1)) {
            children = newLinkedList(copy());
        }
        if (children == null) {
            children = newLinkedList();
        }
        children.addAll(criterions);
        type = otherType;
        return this;
    }

    private static <E> LinkedList<E> newLinkedList(E... elements) {
        LinkedList<E> list = new LinkedList<E>();
        if (elements != null) {
            Collections.addAll(list, elements);
        }
        return list;
    }


}
