/*
 * Project:  hydroplat-parent
 * Module:   hydroplat-common
 * File:     BuilderImpl.java
 * Modifier: yangxin
 * Modified: 2014-06-11 10:38
 *
 * Copyright (c) 2014 Mapjs All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent
 * or the registration of a utility model, design or code.
 */

package cn.gtmap.egovplat.core.data.dsl;

import cn.gtmap.egovplat.core.data.Order;
import cn.gtmap.egovplat.core.data.Pageable;
import cn.gtmap.egovplat.core.util.ArrayUtils;
import cn.gtmap.egovplat.core.util.Pair;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Range;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import org.springframework.util.CollectionUtils;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:yangxin@gtmap.cn">yangxin</a>
 * @version V1.0, 13-6-21
 */
@SuppressWarnings({"unchecked", "rawtypes"})
final class BuilderImpl implements Builder {

    private static final String SPACE = " ";

    private static final int SELECT = 0;
    private static final int FROM = 1;
    private static final int JOIN = 2;
    private static final int GROUP = 3;
    private static final int ORDER = 4;
    private static final int FIELD = 5;

    private final List[] parts = new List[6];
    private CriterionImpl where = new CriterionImpl();
    private CriterionImpl having = new CriterionImpl();

    private Operation operation = Operation.SELECT;
    private boolean isSql = false;
    private boolean isWhere = true;
    private boolean distinct = false;
    private int offset;
    private int size;

    @Override
    public Builder and(Criterion... criterions) {
        if (isWhere) {
            where.and(criterions);
        } else {
            having.and(criterions);
        }
        return this;
    }

    @Override
    public Builder and(Collection<Criterion> criterions) {
        if (isWhere) {
            where.and(criterions);
        } else {
            having.and(criterions);
        }
        return this;
    }

    @Override
    public Builder or(Criterion... criterions) {
        if (isWhere) {
            where.or(criterions);
        } else {
            having.or(criterions);
        }
        return this;
    }

    @Override
    public Builder or(Collection<Criterion> criterions) {
        if (isWhere) {
            where.or(criterions);
        } else {
            having.or(criterions);
        }
        return this;
    }

    @Override
    public Builder expr(String expr) {
        if (isWhere) {
            where.expr(expr);
        } else {
            having.expr(expr);
        }
        return this;
    }

    @Override
    public Builder expr(String... exprs) {
        if (isWhere) {
            where.expr(exprs);
        } else {
            having.expr(exprs);
        }
        return this;
    }

    @Override
    public Builder expr(Collection<String> exprs) {
        if (isWhere) {
            where.expr(exprs);
        } else {
            having.expr(exprs);
        }
        return this;
    }

    @Override
    public Builder isNull(String name) {
        if (isWhere) {
            where.isNull(name);
        } else {
            having.isNull(name);
        }
        return this;
    }

    @Override
    public Builder notNull(String name) {
        if (isWhere) {
            where.notNull(name);
        } else {
            having.notNull(name);
        }
        return this;
    }

    @Override
    public Builder empty(String name) {
        if (isWhere) {
            where.empty(name);
        } else {
            having.empty(name);
        }
        return this;
    }

    @Override
    public Builder notEmpty(String name) {
        if (isWhere) {
            where.notEmpty(name);
        } else {
            having.notEmpty(name);
        }
        return this;
    }

    @Override
    public Builder eq(String name, Object value) {
        if (isWhere) {
            where.eq(name, value);
        } else {
            having.eq(name, value);
        }
        return this;
    }

    @Override
    public Builder eqIf(String name, Object value, boolean... conditions) {
        if (isWhere) {
            where.eqIf(name, value, conditions);
        } else {
            having.eqIf(name, value, conditions);
        }
        return this;
    }

    @Override
    public Builder eqIfHasValue(String name, Object value) {
        if (isWhere) {
            where.eqIfHasValue(name, value);
        } else {
            having.eqIfHasValue(name, value);
        }
        return this;
    }

    @Override
    public Builder eqOrNull(String name, Object value) {
        if (isWhere) {
            where.eqOrNull(name, value);
        } else {
            having.eqOrNull(name, value);
        }
        return this;
    }

    @Override
    public Builder ne(String name, Object value) {
        if (isWhere) {
            where.ne(name, value);
        } else {
            having.ne(name, value);
        }
        return this;
    }

    @Override
    public Builder neIf(String name, Object value) {
        if (isWhere) {
            where.neIf(name, value);
        } else {
            having.neIf(name, value);
        }
        return this;
    }

    @Override
    public Builder neOrNotNull(String name, Object value) {
        if (isWhere) {
            where.neOrNotNull(name, value);
        } else {
            having.neOrNotNull(name, value);
        }
        return this;
    }

    @Override
    public Builder like(String name, String value) {
        if (isWhere) {
            where.like(name, value);
        } else {
            having.like(name, value);
        }
        return this;
    }

    @Override
    public Builder likeIf(String name, String value) {
        if (isWhere) {
            where.likeIf(name, value);
        } else {
            having.likeIf(name, value);
        }
        return this;
    }

    @Override
    public Builder like(String name, String value, MatchMode matchMode) {
        if (isWhere) {
            where.like(name, value, matchMode);
        } else {
            having.like(name, value, matchMode);
        }
        return this;
    }

    @Override
    public Builder likeIf(String name, String value, MatchMode matchMode) {
        if (isWhere) {
            where.likeIf(name, value, matchMode);
        } else {
            having.likeIf(name, value, matchMode);
        }
        return this;
    }

    @Override
    public Builder gt(String name, Object value) {
        if (isWhere) {
            where.gt(name, value);
        } else {
            having.gt(name, value);
        }
        return this;
    }

    @Override
    public Builder gtIf(String name, Object value) {
        if (isWhere) {
            where.gtIf(name, value);
        } else {
            having.gtIf(name, value);
        }
        return this;
    }

    @Override
    public Builder lt(String name, Object value) {
        if (isWhere) {
            where.lt(name, value);
        } else {
            having.lt(name, value);
        }
        return this;
    }

    @Override
    public Builder ltIf(String name, Object value) {
        if (isWhere) {
            where.ltIf(name, value);
        } else {
            having.ltIf(name, value);
        }
        return this;
    }

    @Override
    public Builder ge(String name, Object value) {
        if (isWhere) {
            where.ge(name, value);
        } else {
            having.ge(name, value);
        }
        return this;
    }

    @Override
    public Builder geIf(String name, Object value) {
        if (isWhere) {
            where.geIf(name, value);
        } else {
            having.geIf(name, value);
        }
        return this;
    }

    @Override
    public Builder le(String name, Object value) {
        if (isWhere) {
            where.le(name, value);
        } else {
            having.le(name, value);
        }
        return this;
    }

    @Override
    public Builder leIf(String name, Object value) {
        if (isWhere) {
            where.leIf(name, value);
        } else {
            having.leIf(name, value);
        }
        return this;
    }

    @Override
    public Builder between(String name, Object lo, Object hi) {
        if (isWhere) {
            where.between(name, lo, hi);
        } else {
            having.between(name, lo, hi);
        }
        return this;
    }

    @Override
    public Builder between(String name, Range range) {
        if (isWhere) {
            where.between(name, range);
        } else {
            having.between(name, range);
        }
        return this;
    }

    @Override
    public Builder in(String name, Object value) {
        if (isWhere) {
            where.in(name, value);
        } else {
            having.in(name, value);
        }
        return this;
    }

    @Override
    public Builder in(String name, Object... values) {
        if (isWhere) {
            where.in(name, values);
        } else {
            having.in(name, values);
        }
        return this;
    }

    @Override
    public Builder in(String name, Collection values) {
        if (isWhere) {
            where.in(name, values);
        } else {
            having.in(name, values);
        }
        return this;
    }

    @Override
    public Builder notIn(String name, Object value) {
        if (isWhere) {
            where.notIn(name, value);
        } else {
            having.notIn(name, value);
        }
        return this;
    }

    @Override
    public Builder notIn(String name, Object... values) {
        if (isWhere) {
            where.notIn(name, values);
        } else {
            having.notIn(name, values);
        }
        return this;
    }

    @Override
    public Builder notIn(String name, Collection values) {
        if (isWhere) {
            where.notIn(name, values);
        } else {
            having.notIn(name, values);
        }
        return this;
    }

    @Override
    public Builder exists(Object value) {
        if (isWhere) {
            where.exists(value);
        } else {
            having.exists(value);
        }
        return this;
    }

    @Override
    public Builder notExists(Object value) {
        if (isWhere) {
            where.notExists(value);
        } else {
            having.notExists(value);
        }
        return this;
    }

    @Override
    public Builder sql() {
        isSql = true;
        return this;
    }

    @Override
    public boolean isSql() {
        return isSql;
    }

    @Override
    public Builder from(Class type) {
        return from(type.getSimpleName());
    }

    @Override
    public Builder from(String type) {
        return append(part(FROM), type);
    }

    @Override
    public Builder from(String... types) {
        return from(ArrayUtils.asList(types));
    }

    @Override
    public Builder from(Collection<String> types) {
        return append(part(FROM), types);
    }

    @Override
    public Builder from(Class type, String alias) {
        return from(type.getSimpleName(), alias);
    }

    @Override
    public Builder from(String type, String alias) {
        return append(part(FROM), type + (isSql ? SPACE : " as ") + alias);
    }

    @Override
    public Builder from(Builder subQuery) {
        return append(part(FROM), subQuery);
    }

    @Override
    public Builder from(Builder subQuery, String alias) {
        return append(part(FROM), new Pair<Builder, String>(subQuery, alias));
    }

    @Override
    public Builder insert() {
        operation = Operation.INSERT;
        return sql();
    }

    @Override
    public Builder insert(String type) {
        return insert().from(type);
    }

    @Override
    public Builder update() {
        operation = Operation.UPDATE;
        return this;
    }

    @Override
    public Builder update(Class type) {
        return update(type.getSimpleName());
    }

    @Override
    public Builder update(String type) {
        return update().from(type);
    }

    @Override
    public Builder delete() {
        operation = Operation.DELETE;
        return this;
    }

    @Override
    public Builder delete(Class type) {
        return delete(type.getSimpleName());
    }

    @Override
    public Builder delete(String type) {
        return delete().from(type);
    }

    @Override
    public Builder field(String name, Object value) {
        return append(part(FIELD), new Pair<String, Object>(name, value));
    }

    @Override
    public Builder fields(Object... pairs) {
        if (pairs != null) {
            for (int i = 0, len = pairs.length; i < len; i += 2) {
                Object key = pairs[i];
                if (key instanceof String) {
                    field((String) key, pairs[i + 1]);
                } else {
                    throw new IllegalArgumentException("Key must be string");
                }
            }
        }
        return this;
    }

    @Override
    public Builder fields(Map<String, Object> values) {
        return append(part(FIELD), values.entrySet());
    }

    @Override
    public Builder select(String... fields) {
        return select(ArrayUtils.asList(fields));
    }

    @Override
    public Builder select(Collection<String> fields) {
        return append(part(SELECT), fields);
    }

    @Override
    public Builder select(Builder subQuery) {
        return append(part(SELECT), subQuery);
    }

    @Override
    public Builder select(Builder subQuery, String alias) {
        return append(part(SELECT), new Pair<Builder, String>(subQuery, alias));
    }

    @Override
    public Builder distinct() {
        distinct = true;
        return this;
    }

    @Override
    public Builder join(String path) {
        return join(path, JoinType.INNER);
    }

    @Override
    public Builder join(String path, String alias) {
        return join(path, alias, JoinType.INNER);
    }

    @Override
    public Builder fetchJoin(String path) {
        return fetchJoin(path, JoinType.INNER);
    }

    @Override
    public Builder fetchJoin(String path, String alias) {
        return fetchJoin(path, alias, JoinType.INNER);
    }

    @Override
    public Builder join(String path, JoinType type) {
        return append(part(JOIN), type.getJoinString() + SPACE + path);
    }

    @Override
    public Builder join(String path, String alias, JoinType type) {
        return join(path + (isSql ? SPACE : " as ") + alias, type);
    }

    @Override
    public Builder fetchJoin(String path, JoinType type) {
        return append(part(JOIN), type.getJoinString() + " fetch " + path);
    }

    @Override
    public Builder fetchJoin(String path, String alias, JoinType type) {
        return fetchJoin(path + " as " + alias, type);
    }

    @Override
    public Builder on(String onString) {
        return append(part(JOIN), "on " + onString);
    }

    @Override
    public Builder on(String path1, String field1, String path2, String field2) {
        return on(path1 + "." + field1 + "=" + path2 + "." + field2);
    }

    @Override
    public Builder where(Criterion... criterions) {
        return where(ArrayUtils.asList(criterions));
    }

    @Override
    public Builder where(Collection<Criterion> criterions) {
        isWhere = true;
        where.withCriterions(null, criterions);
        return this;
    }

    @Override
    public Builder group(String... fields) {
        return append(part(GROUP), ArrayUtils.asList(fields));
    }

    @Override
    public Builder having(Criterion... criterions) {
        return having(ArrayUtils.asList(criterions));
    }

    @Override
    public Builder having(Collection<Criterion> criterions) {
        isWhere = false;
        having.withCriterions(null, criterions);
        return this;
    }

    @Override
    public Builder index(int index) {
        if (index > 0) {
            offset = (index - 1) * size;
        }
        return this;
    }

    @Override
    public Builder offset(int offset) {
        if (offset > -1) {
            this.offset = offset;
        }
        return this;
    }

    @Override
    public Builder size(int size) {
        if (size > 0) {
            this.size = size;
        }
        return this;
    }

    @Override
    public Builder with(Pageable pageable) {
        return offset(pageable.getOffset()).size(pageable.getSize()).order(pageable.getOrders());
    }

    @Override
    public Builder desc(String field) {
        return order(Order.desc(field));
    }

    @Override
    public Builder asc(String field) {
        return order(Order.asc(field));
    }

    @Override
    public Builder order(boolean asc, String... fields) {
        return order(Order.orders(asc, fields));
    }

    @Override
    public Builder order(Order... orders) {
        return order(ArrayUtils.asList(orders));
    }

    @Override
    public Builder order(Collection<Order> orders) {
        if (!CollectionUtils.isEmpty(orders))
            for (Order order : orders) {
                append(part(ORDER), order.toString());
            }

        return this;
    }

    @Override
    public QueryParam build() {
        StringBuilder query = new StringBuilder(256);
        StringBuilder countQuery = null;
        if (operation == Operation.SELECT) {
            countQuery = new StringBuilder(256);
        }
        Map<String, Object> args = Maps.newHashMap();
        build(query, countQuery, args, Maps.<String, AtomicInteger>newHashMap());
        return new QueryImpl(args, query.toString(), countQuery == null ? null : countQuery.toString());
    }

    private List part(int partId) {
        List part = parts[partId];
        if (part != null) {
            return part;
        }
        return parts[partId] = Lists.newLinkedList();
    }

    private Builder append(List list, Object... parts) {
        return append(list, ArrayUtils.asList(parts));
    }

    private Builder append(List list, Collection parts) {
        if (isNotEmpty(parts)) {
            list.addAll(parts);
        }
        return this;
    }

    @SuppressWarnings("incomplete-switch")
    private void build(StringBuilder query, StringBuilder countQuery, Map<String, Object> args,
                       Map<String, AtomicInteger> argsIndex) {
        switch (operation) {
            case SELECT:
                List selects = parts[SELECT];
                if (!isNotEmpty(selects)) {
                    if (isSql) {
                        query.append("select * from");
                    } else {
                        query.append("from");
                    }
                    if (countQuery != null) {
                        countQuery.append("select count(*) from");
                    }
                } else {
                    StringBuilder selectSb = new StringBuilder(64);
                    buildList(selects, selectSb, args, argsIndex, false);
                    query.append("select ");
                    if (distinct) {
                        query.append("distinct ");
                    }
                    query.append(selectSb).append(" from");
                    if (countQuery != null) {
                        countQuery.append("select ");
                        countQuery.append("count(").append(distinct ? "distinct " : "").append(selectSb).append(") from");
                    }
                }
                break;
            case UPDATE:
                query.append("update");
                break;
            case DELETE:
                query.append("delete from");
                break;
            case INSERT:
                query.append("insert into");
                break;
        }
        query.append(SPACE);
        StringBuilder fromSb = new StringBuilder(64);
        buildList(parts[FROM], fromSb, args, argsIndex, !isSql && !CollectionUtils.isEmpty(parts[JOIN]));
        query.append(fromSb);
        if (countQuery != null) {
            countQuery.append(SPACE).append(fromSb);
        }
        List<Pair<String, Object>> fields;
        switch (operation) {
            case UPDATE:
                query.append(" set ");
                fields = parts[FIELD];
                for (int i = 0, len = fields.size(); i < len; i++) {
                    Map.Entry<String, Object> pair = fields.get(i);
                    if (i > 0) {
                        query.append(",");
                    }
                    query.append(pair.getKey()).append("=").append(putArg(pair.getKey(), pair.getValue(), args, argsIndex));
                }
                break;
            case INSERT:
                query.append(" (");
                fields = parts[FIELD];
                for (int i = 0, len = fields.size(); i < len; i++) {
                    if (i > 0) {
                        query.append(",");
                    }
                    query.append(fields.get(i).getKey());
                }
                query.append(") values (");
                for (int i = 0, len = fields.size(); i < len; i++) {
                    Map.Entry<String, Object> pair = fields.get(i);
                    if (i > 0) {
                        query.append(",");
                    }
                    query.append(putArg(pair.getKey(), pair.getValue(), args, argsIndex));
                }
                query.append(")");
                break;
        }
        if (operation == Operation.SELECT) {
            List<String> joins = parts[JOIN];
            if (isNotEmpty(joins)) {
                StringBuilder joinSb = new StringBuilder(64);
                joinSb.append(SPACE);
                join(joinSb, joins, SPACE);
                query.append(joinSb);
                if (countQuery != null) {
                    countQuery.append(joinSb);
                }
            }
        }
        if (where != null && where.type != null) {
            switch (operation) {
                case SELECT:
                case UPDATE:
                case DELETE:
                    StringBuilder whereSb = new StringBuilder(128);
                    whereSb.append(" where ");
                    buildCriterion(where, whereSb, args, argsIndex, true);
                    query.append(whereSb);
                    if (countQuery != null) {
                        countQuery.append(whereSb);
                    }
                    break;
            }
        }
        if (operation == Operation.SELECT) {
            List<String> groups = parts[GROUP];
            if (isNotEmpty(groups)) {
                query.append(" group by ");
                join(query, groups, ",");
            }
            if (having != null && having.type != null) {
                query.append(" having ");
                buildCriterion(having, query, args, argsIndex, true);
            }
            List<String> orders = parts[ORDER];
            if (isNotEmpty(orders)) {
                query.append(" order by ");
                join(query, orders, ",");
            }
        }
    }

    private static void join(StringBuilder sb, Collection collection, String separator) {
        boolean isFirst = true;
        for (Object obj : collection) {
            if (isFirst) {
                isFirst = false;
            } else {
                sb.append(separator);
            }
            sb.append(obj);
        }
    }

    private static boolean isNotEmpty(Collection collection) {
        return collection != null && collection.size() > 0;
    }

    private void buildCriterion(CriterionImpl criterion, StringBuilder query, Map<String, Object> args,
                                Map<String, AtomicInteger> argsIndex, boolean isTop) {
        if (criterion.type == null) {
            criterion.type = CriterionImpl.Type.AND;
        }
        switch (criterion.type) {
            case AND:
            case OR:
                if (!isTop) {
                    query.append("(");
                }
                Collection<Criterion> children = criterion.children;
                if (children.size() == 1) {
                    buildCriterion((CriterionImpl) children.iterator().next(), query, args, argsIndex, true);
                } else {
                    int i = 0;
                    for (Criterion child : children) {
                        if (i++ > 0) {
                            query.append(criterion.type == CriterionImpl.Type.AND ? " and " : " or ");
                        }
                        buildCriterion((CriterionImpl) child, query, args, argsIndex, false);
                    }
                }
                if (!isTop) {
                    query.append(")");
                }
                return;
            default:
                break;
        }
        query.append(criterion.name);
        switch (criterion.type) {
            case EXPR:
                return;
            case NULL:
                query.append(" is null");
                return;
            case NOT_NULL:
                query.append(" is not null");
                return;
            default:
                break;
        }
        switch (criterion.type) {
            case BETWEEN:
                Object[] objs = (Object[]) criterion.value;
                query.append(" between ");
                query.append(putArg(criterion.name, objs[0], args, argsIndex));
                query.append(" and ");
                query.append(putArg(criterion.name, objs[1], args, argsIndex));
                return;
            case IN:
                query.append(" in ");
                break;
            case NOT_IN:
                query.append(" not in ");
                break;
            case EXISTS:
                query.append("exists ");
                break;
            case NOT_EXISTS:
                query.append("not exists ");
                break;
            default:
                break;
        }
        switch (criterion.type) {
            case IN:
            case NOT_IN:
            case EXISTS:
            case NOT_EXISTS:
                Object value = criterion.value;
                if (value instanceof Builder) {
                    query.append("(");
                    ((BuilderImpl) value).build(query, null, args, argsIndex);
                } else if (value instanceof Pair) {
                    Pair<String, BuilderImpl> pair = (Pair) value;
                    query.append(pair.getKey());
                    query.append("(");
                    pair.getValue().build(query, null, args, argsIndex);
                    query.append(")");
                } else {
                    query.append("(");
                    if (isSql && value instanceof Iterable) {
                        int i = 0;
                        for (Object v : (Iterable) value) {
                            if (i++ > 0) {
                                query.append(",");
                            }
                            query.append(putArg(criterion.name, v, args, argsIndex));
                        }
                    } else {
                        query.append(putArg(criterion.name, value, args, argsIndex));
                    }
                }
                query.append(")");
                return;
            default:
                break;
        }
        switch (criterion.type) {
            case EQ:
                query.append("=");
                break;
            case NE:
                query.append("<>");
                break;
            case LIKE:
                query.append(" like ");
                break;
            case GT:
                query.append(">");
                break;
            case LT:
                query.append("<");
                break;
            case GE:
                query.append(">=");
                break;
            case LE:
                query.append("<=");
                break;
            default:
                break;
        }
        Object value = criterion.value;
        if (value instanceof Builder) {
            query.append("(");
            ((BuilderImpl) value).build(query, null, args, argsIndex);
            query.append(")");
        } else if (value instanceof Pair) {
            Pair<String, BuilderImpl> pair = (Pair) value;
            query.append(pair.getKey());
            query.append("(");
            pair.getValue().build(query, null, args, argsIndex);
            query.append(")");
        } else {
            query.append(putArg(criterion.name, value, args, argsIndex));
        }
    }

    private String putArg(String key, Object value, Map<String, Object> args, Map<String, AtomicInteger> argsIndex) {
        String uniqueKey = StringUtils.replace(key, ".", "_");
        if (args.containsKey(uniqueKey)) {
            AtomicInteger index = argsIndex.get(key);
            if (index == null) {
                argsIndex.put(key, index = new AtomicInteger());
            }
            uniqueKey += "_" + index.incrementAndGet();
        }
        args.put(uniqueKey, value);
        return ":" + uniqueKey;
    }

    private void buildList(List list, StringBuilder query, Map<String, Object> args, Map<String, AtomicInteger> argsIndex, boolean needRootAlias) {
        for (int i = 0, len = list.size(); i < len; i++) {
            Object item = list.get(i);
            if (i > 0) {
                query.append(",");
            }
            if (item instanceof Builder) {
                query.append("(");
                ((BuilderImpl) item).build(query, null, args, argsIndex);
                query.append(")");
            } else if (item instanceof Pair) {
                Pair<BuilderImpl, String> pair = (Pair) item;
                query.append("(");
                pair.getKey().build(query, null, args, argsIndex);
                query.append(") ");
                query.append(pair.getValue());
            } else {
                if (needRootAlias && i == 0) {
                    String s = item.toString();
                    if (!s.contains(SPACE)) {
                        query.append(s).append(SPACE).append(StringUtils.uncapitalize(s));
                    } else {
                        query.append(StringUtils.replace(s, "as ", ""));
                    }
                } else {
                    query.append(item);
                }
            }
        }
    }

    static enum Operation {
        SELECT, UPDATE, DELETE, INSERT
    }

    class QueryImpl implements QueryParam {

        private final Map<String, Object> args;
        private final String query;
        private final String countQuery;

        QueryImpl(Map<String, Object> args, String query, String countQuery) {
            this.args = args;
            this.query = query;
            this.countQuery = countQuery;
        }

        @Override
        public boolean isSql() {
            return isSql;
        }

        @Override
        public Map<String, Object> getArgs() {
            return args;
        }

        @Override
        public String getQuery() {
            return query;
        }

        @Override
        public String getCountQuery() {
            return countQuery;
        }

        @Override
        public int getOffset() {
            return offset;
        }

        @Override
        public int getSize() {
            return size;
        }

        public String toString() {
            return ToStringBuilder.reflectionToString(this, ToStringStyle.MULTI_LINE_STYLE);
        }
    }

}
