/*
 * Project:  hydroplat-parent
 * Module:   hydroplat-common
 * File:     AbstractAttrable.java
 * Modifier: yangxin
 * Modified: 2014-06-11 10:38
 *
 * Copyright (c) 2014 Mapjs All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent
 * or the registration of a utility model, design or code.
 */

package cn.gtmap.egovplat.core.attr;

import cn.gtmap.egovplat.core.bean.Attrable;
import cn.gtmap.egovplat.core.util.TypeReference;
import com.alibaba.fastjson.annotation.JSONField;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import cn.gtmap.egovplat.core.Constants;
import cn.gtmap.egovplat.core.util.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.util.PropertyPlaceholderHelper;

import java.util.*;

import static com.alibaba.fastjson.util.TypeUtils.*;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oznyang@163.com">oznyang</a>
 * @version V1.0, 13-10-31
 */
public abstract class AbstractAttrable implements Attrable {

    @Override
    public boolean isAttrsEmpty() {
        return getAttrKeys().size() == 0;
    }

    @Override
    public boolean hasAttr(String key) {
        return getAttr(key) != null;
    }

    @Override
    public String getAttr(String key) {
        return getAttr(key, String.class);
    }

    @Override
    public String getAttr(String key, String defaultValue) {
        String value = getAttr(key);
        return value == null ? defaultValue : value;
    }

    @Override
    public String getRequiredAttr(String key) throws IllegalStateException {
        String value = getAttr(key);
        if (value == null) {
            throw new IllegalStateException("required key [" + key + "] not found");
        }
        return value;
    }

    @Override
    public String getExprAttr(String text) throws IllegalStateException {
        return Constants.PLACEHOLDER_HELPER.replacePlaceholders(text, new PropertyPlaceholderHelper.PlaceholderResolver() {
            @Override
            public String resolvePlaceholder(String placeholderName) {
                return getAttr(placeholderName);
            }
        });
    }

    @Override
    public boolean getBoolAttr(String key) {
        String value = getAttr(key);
        return StringUtils.isEmpty(value) ? false : castToBoolean(value);
    }

    @Override
    public byte getByteAttr(String key) {
        String value = getAttr(key);
        return StringUtils.isEmpty(value) ? 0 : castToByte(value);
    }

    @Override
    public short getShortAttr(String key) {
        String value = getAttr(key);
        return StringUtils.isEmpty(value) ? 0 : castToShort(value);
    }

    @Override
    public int getIntAttr(String key) {
        String value = getAttr(key);
        return StringUtils.isEmpty(value) ? 0 : castToInt(value);
    }

    @Override
    public long getLongAttr(String key) {
        String value = getAttr(key);
        return StringUtils.isEmpty(value) ? 0L : castToLong(value);
    }

    @Override
    public float getFloatAttr(String key) {
        String value = getAttr(key);
        return StringUtils.isEmpty(value) ? 0F : castToFloat(value);
    }

    @Override
    public double getDoubleAttr(String key) {
        String value = getAttr(key);
        return StringUtils.isEmpty(value) ? 0D : castToDouble(value);
    }

    @Override
    public byte[] getBytesAttr(String key) {
        String value = getAttr(key);
        return StringUtils.isEmpty(value) ? ArrayUtils.EMPTY_BYTE_ARRAY : castToBytes(value);
    }

    @Override
    public Date getDateAttr(String key) {
        String value = getAttr(key);
        return castToDate(value);
    }

    @Override
    public <T> T getAttr(String key, Class<T> targetType, T defaultValue) {
        T value = getAttr(key, targetType);
        return value == null ? defaultValue : value;
    }

    @Override
    public <T> T getRequiredAttr(String key, Class<T> targetType) throws IllegalStateException {
        T value = getAttr(key, targetType);
        if (value == null) {
            throw new IllegalStateException("required key [" + key + "] not found");
        }
        return value;
    }

    @Override
    public <T> T getAttr(String key, TypeReference<T> targetType) {
        return getAttr(key, targetType.getType());
    }

    @Override
    public <T> T getAttr(String key, TypeReference<T> targetType, T defaultValue) {
        return getAttr(key, targetType.getType(), defaultValue);
    }

    @Override
    public <T> T getRequiredAttr(String key, TypeReference<T> targetType) throws IllegalStateException {
        return getRequiredAttr(key, targetType.getType());
    }

    @Override
    public String[] getArrayAttr(String key) {
        return getAttr(key, String[].class);
    }

    @Override
    @JSONField(serialize = false)
    public Map<String, String> getAttrs() {
        return getAttrs(Collections.<String>emptyList());
    }

    @Override
    public Map<String, String> getAttrs(String... keys) {
        return getAttrs(ArrayUtils.asList(keys));
    }

    @Override
    public Map<String, String> getAttrs(Collection<String> keys) {
        if (keys == null || keys.isEmpty()) {
            keys = getAttrKeys();
        }
        Map<String, String> map = Maps.newHashMapWithExpectedSize(keys.size());
        for (String key : keys) {
            String value = getAttr(key);
            if (value != null) {
                map.put(key, value);
            }
        }
        return Collections.unmodifiableMap(map);
    }

    @Override
    public void setAttrs(Map<String, ?> map) {
        for (Map.Entry<String, ?> entry : map.entrySet()) {
            setAttr(entry.getKey(), entry.getValue());
        }
    }

    @Override
    public void removeAttrs(String... keys) {
        removeAttrs(ArrayUtils.asList(keys));
    }

    @Override
    public void removeAttrs(Collection<String> keys) {
        for (String key : keys) {
            removeAttr(key);
        }
    }

    @Override
    public void clearAttrs() {
        removeAttrs(Lists.newArrayList(getAttrKeys()));
    }

    @Override
    public String toString() {
        return getAttrs().toString();
    }
}
