/*
 * Project:  hydroplat-parent
 * Module:   hydroplat-common
 * File:     RequestUtils.java
 * Modifier: yangxin
 * Modified: 2014-06-11 21:48
 *
 * Copyright (c) 2014 Mapjs All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent
 * or the registration of a utility model, design or code.
 */
package cn.gtmap.common.utils;

import cn.gtmap.common.core.ex.AppException;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.gtis.config.AppConfig;
import com.gtis.util.ThreadPool;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpConnectionManager;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.apache.commons.httpclient.NameValuePair;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.params.HttpMethodParams;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.springframework.util.AntPathMatcher;
import org.springframework.util.PathMatcher;
import org.springframework.util.PropertyPlaceholderHelper;
import org.springframework.web.util.UrlPathHelper;

import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.SocketTimeoutException;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:yangxin@gtmap.cn">yangxin</a>
 * @version V1.0, 11-12-2
 */
public final class RequestUtils {

    private static final Pattern QUERY_PARAM_PATTERN = Pattern.compile("([^&=]+)(=?)([^&]+)?");

    public static String getParameter(HttpServletRequest request, String name) {
        String query = request.getQueryString();
        if (query != null) {
            Matcher m = QUERY_PARAM_PATTERN.matcher(query);
            while (m.find()) {
                if (name.equals(m.group(1))) {
                    return m.group(3);
                }
            }
        }
        return null;
    }

    public static UrlPathHelper URL_PATH_HELPER = new UrlPathHelper() {
        @Override
        public String getLookupPathForRequest(HttpServletRequest request) {
            String key = request.getRequestURI() + "_lookupPath";
            String path = (String) request.getAttribute(key);
            if (path == null) {
                request.setAttribute(key, path = super.getPathWithinApplication(request));
            }
            return path;
        }
    };

    public static PathMatcher PATH_MATCHER = new AntPathMatcher();

    public static String getClientIP(HttpServletRequest request) {
        String xForwardedFor;
        xForwardedFor = StringUtils.trimToNull(request.getHeader("$wsra"));
        if (xForwardedFor != null) {
            return xForwardedFor;
        }
        xForwardedFor = StringUtils.trimToNull(request.getHeader("X-Real-IP"));
        if (xForwardedFor != null) {
            return xForwardedFor;
        }
        xForwardedFor = StringUtils.trimToNull(request.getHeader("X-Forwarded-For"));
        if (xForwardedFor != null) {
            int spaceIndex = xForwardedFor.indexOf(',');
            if (spaceIndex > 0) {
                return xForwardedFor.substring(0, spaceIndex);
            } else {
                return xForwardedFor;
            }
        }
        return request.getRemoteAddr();
    }

    /**
     * 获取请求主机IP地址,如果通过代理进来，则透过防火墙获取真实IP地址;
     *
     * @param request
     * @return
     */
    public static String getIpAddress(HttpServletRequest request){
        // 获取请求主机IP地址,如果通过代理进来，则透过防火墙获取真实IP地址
        String ip = request.getHeader("X-Forwarded-For");
        if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
                ip = request.getHeader("Proxy-Client-IP");
            }
            if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
                ip = request.getHeader("WL-Proxy-Client-IP");
            }
            if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
                ip = request.getHeader("HTTP_CLIENT_IP");
            }
            if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
                ip = request.getHeader("HTTP_X_FORWARDED_FOR");
            }
            if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
                ip = request.getRemoteAddr();
            }
        } else if (ip.length() > 15) {
            String[] ips = ip.split(",");
            for (int index = 0; index < ips.length; index++) {
                String strIp = (String) ips[index];
                if (!("unknown".equalsIgnoreCase(strIp))) {
                    ip = strIp;
                    break;
                }
            }
        }
        return ip;
    }

    public static String getDomain(HttpServletRequest request) {
        StringBuffer url = request.getRequestURL();
        int end = url.indexOf(".");
        if (end == -1)
            return "";
        int start = url.indexOf("//");
        return url.substring(start + 2, end);
    }

    public static String toJSON(Object obj) {
        return JSON.toJSONString(obj, SerializerFeature.DisableCircularReferenceDetect);
    }

    public static String formatUrl(String base, String path) {
        StringBuilder sb = new StringBuilder();
        if (!"/".equals(base)) {
            sb.append(base);
        }
        if (path.charAt(0) != '/') {
            sb.append("/");
        }
        sb.append(path);
        return sb.toString();
    }

    public static String buildUrl(String path, Map<?, ?> args) {
        if (StringUtils.isEmpty(path)) {
            return "";
        }
        String url = path.startsWith("/") ? "" : path;
        StringBuilder sb = new StringBuilder(url);
        boolean isFirst = true;
        if (args != null) {
            for (Map.Entry<?, ?> entry : args.entrySet()) {
                if (isFirst) {
                    isFirst = false;
                    sb.append(url.contains("?") ? "&" : "?");
                } else {
                    sb.append("&");
                }
                String key = encode(entry.getKey());
                Object value = entry.getValue();
                if (value == null) {
                    sb.append(key);
                } else if (value instanceof Iterable) {
                    boolean isFirst1 = true;
                    for (Object value1 : (Iterable) value) {
                        if (isFirst1) {
                            isFirst1 = false;
                        } else {
                            sb.append("&");
                        }
                        sb.append(key).append("=").append(encode(value1));
                    }
                } else if (value.getClass().isArray()) {
                    boolean isFirst1 = true;
                    for (Object value1 : (Object[]) value) {
                        if (isFirst1) {
                            isFirst1 = false;
                        } else {
                            sb.append("&");
                        }
                        sb.append(key).append("=").append(encode(value1));
                    }
                } else {
                    sb.append(key).append("=").append(encode(value));
                }
            }
        }
        return sb.toString();
    }

    public static String encode(Object url) {
        try {
            return URLEncoder.encode(url.toString(), Charsets.UTF8);
        } catch (UnsupportedEncodingException e) {
            throw new AppException(e);
        }
    }

    public static String decode(String url) {
        try {
            return URLEncoder.encode(url, Charsets.UTF8);
        } catch (UnsupportedEncodingException e) {
            throw new AppException(e);
        }
    }

    public static boolean isPost(HttpServletRequest request) {
        return "POST".equals(request.getMethod());
    }

    public static boolean matchAny(HttpServletRequest request, UrlPathHelper urlPathHelper, PathMatcher pathMatcher, String[] patterns) {
        if (ArrayUtils.isNotEmpty(patterns)) {
            String lookupPath = urlPathHelper.getLookupPathForRequest(request);
            for (String pattern : patterns) {
                if (pathMatcher.match(pattern, lookupPath)) {
                    return true;
                }
            }
        }
        return false;
    }

    public static boolean matchAll(HttpServletRequest request, UrlPathHelper urlPathHelper, PathMatcher pathMatcher, String[] patterns) {
        if (ArrayUtils.isNotEmpty(patterns)) {
            String lookupPath = urlPathHelper.getLookupPathForRequest(request);
            for (String pattern : patterns) {
                if (!pathMatcher.match(pattern, lookupPath)) {
                    return false;
                }
            }
        }
        return true;
    }

    public static String getString(HttpServletRequest request, String name) {
        return request.getParameter(name);
    }

    public static String getString(HttpServletRequest request, String name, String def) {
        String value = getString(request, name);
        return StringUtils.isEmpty(value) ? def : value;
    }

    public static Boolean getBool(HttpServletRequest request, String name) {
        String value = request.getParameter(name);
        return StringUtils.isEmpty(value) ? null : BooleanUtils.toBooleanObject(value);
    }

    public static boolean getBool(HttpServletRequest request, String name, boolean def) {
        Boolean value = getBool(request, name);
        return value == null ? def : value;
    }

    public static Integer getInt(HttpServletRequest request, String name) {
        String value = request.getParameter(name);
        return StringUtils.isEmpty(value) ? null : NumberUtils.createInteger(value);
    }

    public static int getInt(HttpServletRequest request, String name, int def) {
        Integer value = getInt(request, name);
        return value == null ? def : value;
    }

    public static Double getDouble(HttpServletRequest request, String name) {
        String value = request.getParameter(name);
        return StringUtils.isEmpty(value) ? null : NumberUtils.createDouble(value);
    }

    public static double getDouble(HttpServletRequest request, String name, double def) {
        Double value = getDouble(request, name);
        return value == null ? def : value;
    }

    public static String initOptProperties(String url) {
        if (url != null) {
            PropertyPlaceholderHelper propertyPlaceholderHelper = new PropertyPlaceholderHelper("${", "}");
            Properties properties = new Properties();
            properties.putAll(AppConfig.getProperties());
            url = propertyPlaceholderHelper.replacePlaceholders(url, properties);
        }

        return url;
    }

    /**
     * 请求个业务系统数据
     * @param url
     * @return
     */
    public static String httpPostRequest(final String url, final HashMap param, boolean newThread){
        if (newThread){
            ThreadPool.execute(new Runnable() {
                public void run() {
                    httpPostRequest(url,param);
                }
            });
        }else {
            return httpPostRequest(url,param);
        }
        return null;
    }

    public static String httpPostRequest(final String url, final HashMap param){
        String resultStr = null;
        try {
            HttpConnectionManager connectionManager = new MultiThreadedHttpConnectionManager();
            connectionManager.getParams().setDefaultMaxConnectionsPerHost(10);
            connectionManager.getParams().setConnectionTimeout(30000);
            connectionManager.getParams().setSoTimeout(30000);
            HttpClient httpClient = new HttpClient(connectionManager);
            if (StringUtils.isNotBlank(url)) {
                PostMethod postMethod = new PostMethod(url);
                try {
                    postMethod.getParams().setParameter(HttpMethodParams.HTTP_CONTENT_CHARSET, "UTF-8");
                    /*NameValuePair[] postData = new NameValuePair[1];
                    postData[0] = new NameValuePair("proid", proid);*/
                    if (param != null && param.entrySet().size() > 0){
                        NameValuePair[] postData = new NameValuePair[param.entrySet().size()];
                        Iterator iter = param.entrySet().iterator();
                        int i=0;
                        while (iter.hasNext()) {
                            Map.Entry entry = (Map.Entry) iter.next();
                            String key = entry.getKey().toString();
                            String val = entry.getValue()==null?"":entry.getValue().toString();
                            postData[i] = new NameValuePair(key, val);
                            i++;
                        }
                        postMethod.addParameters(postData);
                    }

                    httpClient.executeMethod(postMethod);
                    String info = postMethod.getResponseBodyAsString();
                    if (StringUtils.isNotBlank(info)){
                        System.out.println(info);
                        resultStr = info;
                    }
                } catch (SocketTimeoutException eTimeOut) {
                    eTimeOut.printStackTrace();
                } catch (IOException e) {
                    e.printStackTrace();
                } catch (Exception e) {
                    e.printStackTrace();
                }finally {
                    if(postMethod!=null)
                        postMethod.releaseConnection();
                }
            }
        } catch (Exception e) {
        }
        return resultStr;
    }


}
