/*
 * Project:  hydroplat-parent
 * Module:   hydroplat-common
 * File:     FmUtils.java
 * Modifier: yangxin
 * Modified: 2014-06-25 17:35
 *
 * Copyright (c) 2014 Mapjs All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent
 * or the registration of a utility model, design or code.
 */

package cn.gtmap.common.core.support.freemarker;


import cn.gtmap.common.utils.Charsets;
import cn.gtmap.common.utils.HtmlUtils;
import freemarker.core.Environment;
import freemarker.ext.beans.BeansWrapper;
import freemarker.template.*;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.util.CollectionUtils;

import java.io.IOException;
import java.io.StringWriter;
import java.util.Map;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:yangxin@gtmap.cn">yangxin</a>
 * @version V1.0, 14-6-25
 */
@SuppressWarnings("unchecked")
public final class FmUtils {
    static BeansWrapper wrapper;

    public static <T> T unwrap(Object obj) throws TemplateModelException {
        if (obj == null) {
            return null;
        }
        if (obj instanceof TemplateModel) {
            return (T) wrapper.unwrap((TemplateModel) obj);
        }
        return (T) obj;
    }

    public static TemplateModel wrap(Object obj) throws TemplateModelException {
        if (obj == null) {
            return null;
        }
        if (obj instanceof TemplateModel) {
            return (TemplateModel) obj;
        }
        return wrapper.wrap(obj);
    }

    public static <T> T getVar(Environment env, String key) throws TemplateModelException {
        return unwrap(env.getVariable(key));
    }

    public static <T> T getVar(Environment env, String key, T def) throws TemplateModelException {
        T value = getVar(env, key);
        return value == null ? def : value;
    }

    public static void setVar(Environment env, String key, Object value) throws TemplateModelException {
        env.setVariable(key, wrap(value));
    }

    public static void delVar(Environment env, String... keys) throws TemplateModelException {
        for (String key : keys) {
            env.getCurrentNamespace().remove(key);
        }
    }

    public static <T> T getParam(Map params, String key) throws TemplateModelException {
        TemplateModel model = (TemplateModel) params.get(key);
        if (model == null) {
            return null;
        }
        return unwrap(model);
    }

    public static String getString(Map params, String key) throws TemplateModelException {
        TemplateModel model = (TemplateModel) params.get(key);
        if (model == null) {
            return null;
        }
        if (model instanceof TemplateScalarModel) {
            return ((TemplateScalarModel) model).getAsString();
        } else if ((model instanceof TemplateNumberModel)) {
            return ((TemplateNumberModel) model).getAsNumber().toString();
        } else {
            return model.toString();
        }
    }

    public static Integer getInteger(Map params, String key) throws TemplateModelException {
        TemplateModel model = (TemplateModel) params.get(key);
        if (model == null) {
            return null;
        }
        if ((model instanceof TemplateNumberModel)) {
            return ((TemplateNumberModel) model).getAsNumber().intValue();
        } else {
            try {
                return Integer.valueOf(model instanceof TemplateScalarModel ? ((TemplateScalarModel) model).getAsString() : model.toString());
            } catch (NumberFormatException e) {
                return null;
            }
        }
    }

    public static Boolean getBoolean(Map params, String key) throws TemplateModelException {
        TemplateModel model = (TemplateModel) params.get(key);
        if (model == null) {
            return null;
        }
        if (model instanceof TemplateBooleanModel) {
            return ((TemplateBooleanModel) model).getAsBoolean();
        } else if ((model instanceof TemplateNumberModel)) {
            return ((TemplateNumberModel) model).getAsNumber().intValue() != 0;
        } else {
            return BooleanUtils.toBooleanObject(model instanceof TemplateScalarModel ? ((TemplateScalarModel) model).getAsString() : model.toString());
        }
    }


    public static boolean getBoolean(Map params, String key, boolean def) throws TemplateModelException {
        Boolean value = getBoolean(params, key);
        return value == null ? def : value;
    }

    public static String getString(Map params, String key, String def) throws TemplateModelException {
        String value = getString(params, key);
        return StringUtils.isEmpty(value) ? def : value;
    }

    public static void includeTpl(String tplName, Map<String, Object> params, Environment env) throws IOException, TemplateException {
        if (!CollectionUtils.isEmpty(params)) {
            for (Map.Entry<String, Object> entry : params.entrySet()) {
                env.setVariable(entry.getKey(), wrap(entry.getValue()));
            }
        }
        env.include(tplName, Charsets.UTF8, true);
        if (!CollectionUtils.isEmpty(params)) {
            for (Map.Entry<String, Object> entry : params.entrySet()) {
                delVar(env, entry.getKey());
            }
        }
    }

    public static String renderBody(TemplateDirectiveBody body) throws IOException, TemplateException {
        if (body == null) {
            return null;
        }
        StringWriter sw = new StringWriter();
        body.render(sw);
        return sw.toString();
    }

    public static void append(StringBuilder sb, String key, Object value) {
        if (value != null) {
            sb.append(" ").append(key).append("=\"").append(value).append("\"");
        }
    }

    public static void appendE(StringBuilder sb, String key, Object value) {
        if (value != null) {
            sb.append(" ").append(key).append("=\"").append(HtmlUtils.escape(value)).append("\"");
        }
    }

    public static void appendUE(StringBuilder sb, String key, Object value) throws TemplateModelException {
        if (value != null) {
            sb.append(" ").append(key).append("=\"").append(HtmlUtils.escape(unwrap(value))).append("\"");
        }
    }

    public FmUtils() {
    }
}
