/*
 * Project:  hydroplat-parent
 * Module:   hydroplat-common
 * File:     ExUtils.java
 * Modifier: yangxin
 * Modified: 2014-06-11 10:38
 *
 * Copyright (c) 2014 Mapjs All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent
 * or the registration of a utility model, design or code.
 */

package cn.gtmap.common.utils;

import cn.gtmap.common.core.ex.AppException;
import cn.gtmap.common.core.ex.ErrorCode;
import cn.gtmap.common.core.ex.Source;
import cn.gtmap.common.core.ex.SourceExtractor;
import cn.gtmap.common.core.i18n.NLS;
import com.google.common.collect.Sets;
import freemarker.template.TemplateException;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.TypeMismatchException;
import org.springframework.core.NestedRuntimeException;
import org.springframework.validation.BindException;
import org.springframework.web.bind.MissingServletRequestParameterException;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.lang.reflect.InvocationTargetException;
import java.net.SocketException;
import java.sql.SQLException;
import java.util.*;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:yangxin@gtmap.cn">yangxin</a>
 * @version V1.0, 12-8-12
 */
public final class ExUtils {

    public static Throwable getRootCause(Throwable cause) {
        Throwable rootCause = null;
        while (cause != null && cause != rootCause) {
            rootCause = cause;
            cause = cause.getCause();
        }
        return rootCause;
    }

    public static AppException wrap(Throwable ex) {
        if (ex instanceof InvocationTargetException) {
            ex = ((InvocationTargetException) ex).getTargetException();
        }
        if (ex instanceof AppException) {
            return (AppException) ex;
        }
        return new AppException(ex);
    }

    public static void throwEx(Throwable ex) {
        if (ex instanceof RuntimeException) {
            throw (RuntimeException) ex;
        } else {
            throw wrap(ex);
        }
    }

    public static void throwRoot(Exception e) {
        throwEx(ExUtils.getRootCause(e));
    }

    public static Exception raw(Exception ex) {
        return ex instanceof InvocationTargetException ? (Exception) ((InvocationTargetException) ex).getTargetException() : ex;
    }

    public static int parseCode(Throwable cause) {
        if (cause instanceof InvocationTargetException) {
            cause = ((InvocationTargetException) cause).getTargetException();
        }
        if (cause instanceof AppException) {
            return ((AppException) cause).getCode();
        } else if (cause instanceof NullPointerException) {
            return ErrorCode.NPE_EX;
        } else if (cause instanceof SocketException) {
            return ErrorCode.SOCKET_EX;
        } else if (cause instanceof SQLException) {
            return ErrorCode.DB_EX;
        } else if (cause instanceof IOException) {
            return ErrorCode.IO_EX;
        } else if (cause instanceof TemplateException) {
            return ErrorCode.TPL_EX;
        } else if (cause instanceof ClassNotFoundException || cause instanceof IllegalAccessException) {
            return ErrorCode.CLASS_EX;
        } else if (cause instanceof UnsupportedOperationException) {
            return ErrorCode.UN_SUPPORTED;
        } else if (cause instanceof IllegalStateException) {
            return ErrorCode.ILLEGAL_STATE;
        } else if (cause instanceof IllegalArgumentException) {
            return cause instanceof IllegalFormatException ? ErrorCode.ILLEGAL_ARG_FORMAT : ErrorCode.ILLEGAL_ARG;
        } else if (cause instanceof VirtualMachineError) {
            if (cause instanceof OutOfMemoryError) {
                return ErrorCode.OOM_ERROR;
            } else if (cause instanceof StackOverflowError) {
                return ErrorCode.SOF_ERROR;
            }
        }
        return ErrorCode.SERVER_EX;
    }

    public static String buildMessage(int code, Object[] args, String customMessage, Throwable cause) {
        String message = NLS.get("error." + code, args);

        if (message == null) {
            message = "errorcode:" + code;
        }
        if (StringUtils.isNotEmpty(customMessage)) {
            message = message + ", " + customMessage;
        }
        if (cause != null) {
            message = buildNestedMessage(message, cause);
        }
        return message;
    }

    public static String buildNestedMessage(String message, Throwable ex) {
        StringBuilder sb = new StringBuilder(message);
        Set<Throwable> visitedExceptions = Sets.newHashSet();
        Throwable tmpEx = ex;
        do {
            if (sb.length() > 0) {
                sb.append("; ");
            }
            sb.append(tmpEx);
            visitedExceptions.add(tmpEx);
            tmpEx = tmpEx.getCause();
        }
        while (!(tmpEx == null || visitedExceptions.contains(tmpEx) || tmpEx instanceof AppException || tmpEx instanceof NestedRuntimeException));
        return sb.toString();
    }

    public static String buildNestedMessage(Throwable ex) {
        return buildNestedMessage("", ex);
    }

    public static String buildStackTrace(Throwable cause) {
        StringWriter sw = new StringWriter();
        cause.printStackTrace(new PrintWriter(sw));
        return sw.toString();
    }

    public static Map<String, Object> toMap(Exception ex) {
        int code;
        String msg = null;
        if (ex instanceof AppException) {
            code = ((AppException) ex).getCode();
            msg = ex.getMessage();
        } else if (ex instanceof MissingServletRequestParameterException) {
            code = ErrorCode.MISSING_ARG;
        } else if (ex instanceof TypeMismatchException || ex instanceof BindException) {
            code = ErrorCode.ILLEGAL_ARG;
        } else {
            code = parseCode(ex);
        }
        if (msg == null) {
            msg = buildNestedMessage(ex);
        }
        Map<String, Object> map = new LinkedHashMap<String, Object>(3);
        map.put("ret", code);
        map.put("msg", msg);
//        if (Env.isDev()) {
//            map.put("detail", ExUtils.buildStackTrace(ex));
//        }
        return map;
    }

    @SuppressWarnings("unchecked")
    public static Source extractSource(List<SourceExtractor> sourceExtractors, Throwable ex) {
        for (SourceExtractor se : sourceExtractors) {
            if (se.isSupport(ex)) {
                Source source = se.extract(ex);
                if (source != null) {
                    return source;
                }
            }
        }
        return null;
    }

    private ExUtils() {
    }
}
